
import json

from apispec import APISpec
from apispec.ext.marshmallow import MarshmallowPlugin
from common.backend.schemas.ask import AskRequestSchema, AskResponseSchema
from common.backend.schemas.conversations import ConversationsResponseSchema, SingleConversationResponseSchema

# Create an OpenAPI spec
spec = APISpec(
    title="Answers API",
    version="1.0.0",
    openapi_version="3.0.0",
    plugins=[MarshmallowPlugin()],
)


#################################################
# Schemas
#################################################


# Ask schemas 
spec.components.schema("AskRequest", schema=AskRequestSchema)
spec.components.schema("AskResponse", schema=AskResponseSchema)

# Conversation schemas
spec.components.schema("ConversationResponse", schema=ConversationsResponseSchema)
spec.components.schema("SingleConversationResponse", schema=SingleConversationResponseSchema)


#################################################
# Endpoints
#################################################


# Ask EPs
spec.path(
    path="/ask",
    operations={
        "post": {
            "summary": "Ask a question",
            "description": "Sends a question and receives an answer.",
            "tags": ["Ask"],
            "requestBody": {
                "required": True,
                "content": {"application/json": {"schema": {"$ref": "#/components/schemas/AskRequest"}}},
            },
            "responses": {
                "200": {
                    "description": "Successful response",
                    "content": {"application/json": {"schema": {"$ref": "#/components/schemas/AskResponse"}}},
                },
                "400": {"description": "Bad request"},
            },
            "parameters": [
                {
                    "name": "Content-Type",
                    "in": "header",
                    "required": True,
                    "description": "Used to specify the media type of the request body.",
                    "schema": {
                        "type": "string",
                        "enum": ["application/json"],  
                    },
                },
                {
                    "name": "Accept",
                    "in": "header",
                    "required": True,
                    "description": """Used to specify the media type of the response body. \n
    - `application/json` (default): will return a JSON response\n 
    - `text/event-stream`: Enables streaming response using Server-Sent Events (SSE).
                        """,
                    "schema": {
                        "type": "string",
                        "enum": ["application/json", "text/event-stream"],  
                    },
                },
            ],
        }
    },
)

# Conversations EPs
spec.path(
    path="/conversations",
    operations={
        "get": {
            "summary": "Get user conversations",
            "description": "Get all conversations of an authenticated user",
            "tags": ["Conversations"],
            "parameters": [
                {
                    "name": "applicationId",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The ID of the application",
                },
                {
                    "name": "user",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The user identifier",
                },
            ],
            "responses": {
                "200": {
                    "description": "Successful response",
                    "content": {"application/json": {"schema": {"$ref": "#/components/schemas/ConversationResponse"}}},
                },
                "400": {"description": "Bad request"},
            },
        },
        "delete": {
            "summary": "Delete user conversations",
            "description": "Delete all conversations of an authenticated user",
            "tags": ["Conversations"],
            "parameters": [
                {
                    "name": "applicationId",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The ID of the application",
                },
                {
                    "name": "user",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The user identifier",
                },
            ],
            "responses": {
                "200": {
                    "description": "Successful response",
                    "content": {
                        "application/json": {
                            "example": {
                                "status": "ok",
                                "message": "Conversations deleted successfully"
                            }
                        }
                    }
                },
                "400": {"description": "Bad request"},
            },
        }
    },
)

spec.path(
    path="/conversations/{conversation_id}",
    parameters=[
        {
            "name": "conversation_id",
            "in": "path",
            "required": True,
            "schema": {"type": "string"},
            "description": "The ID of the conversation",
        }
    ],
    operations={
        "get": {
            "summary": "Get a specific conversation",
            "description": "Retrieve a specific conversation by its ID",
            "tags": ["Conversations"],
            "parameters": [
                {
                    "name": "applicationId",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The ID of the application",
                },
                {
                    "name": "user",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The user identifier",
                },
            ],
            "responses": {
                "200": {
                    "description": "Successful response",
                    "content": {
                        "application/json": {
                            "schema": {"$ref": "#/components/schemas/SingleConversationResponse"}
                        }
                    }
                },
                "400": {"description": "Bad request"},
            },
        },
        "delete": {
            "summary": "Delete a specific conversation",
            "description": "Delete a specific conversation by its ID",
            "tags": ["Conversations"],
            "parameters": [
                {
                    "name": "applicationId",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The ID of the application",
                },
                {
                    "name": "user",
                    "in": "query",
                    "required": True,
                    "schema": {"type": "string"},
                    "description": "The user identifier",
                },
            ],
            "responses": {
                "200": {
                    "description": "Successful response",
                    "content": {
                        "application/json": {
                            "example": {
                                "status": "ok",
                                "message": "Conversation deleted successfully"
                            }
                        }
                    }
                },
                "400": {"description": "Bad request"},
            },
        }
    },
)


#################################################
# Generate
#################################################


# Generate OpenAPI spec as a JSON file
with open("openapi.json", "w", encoding="utf-8") as f:
    json.dump(spec.to_dict(), f, indent=2)

# Print output
print("OpenAPI schema generated and saved as openapi.json")