import { marked } from 'marked'
import dompurify from 'dompurify'
import Prism from 'prismjs'

type Options = {
  target: '_blank' | '_self' | '_parent' | '_top'
}

const renderer = new marked.Renderer()

/**
 * Render and sanitize a markdown string
 */
export function md(
  str: string,
  translations: any,
  queryIndex: number,
  options: Options = { target: '_blank' }
): string {
  // Add hook to ensure links have target and rel attributes
  dompurify.addHook('afterSanitizeAttributes', (node) => {
    if (node.tagName === 'A') {
      if (!node.getAttribute('target')) {
        node.setAttribute('target', '_blank')
      }
      if (node.getAttribute('target') === '_blank') {
        node.setAttribute('rel', 'noopener noreferrer')
      }
    }
  })

  // Custom link renderer
  renderer.link = function (href, title, text) {
    const link = marked.Renderer.prototype.link.apply(this, [href, title, text])
    return link.replace('<a', `<a target="${options.target}" rel="noopener noreferrer" class="text-blue-600 underline"`)
  }

  // Paragraph renderer to preserve newlines as <br />
  renderer.paragraph = function (text) {
    return `<p class="my-2">${text.replace(/\n/g, '<br />')}</p>`
  }

  // List renderer for ordered and unordered lists
  renderer.list = function (body, ordered, start) {
    const type = ordered ? 'ol' : 'ul'
    const listClass = ordered ? 'list-decimal pl-6' : 'list-disc pl-6'
    const startAttr = ordered && start !== 1 ? ` start="${start}"` : ''
    return `<${type} class="${listClass} mb-4"${startAttr}>${body}</${type}>`
  }

  // List item renderer
  renderer.listitem = function (text) {
    return `<li class="mb-1">${text}</li>`
  }

  // Blockquote renderer
  renderer.blockquote = function (quote) {
    return `
      <blockquote class="border-l-2 p-2 my-4 rounded-r-md text-gray-700 italic">
        ${quote}
      </blockquote>
    `;
  }

  renderer.codespan = function (text) {
    return `<code class="bg-gray-200 text-gray-800 p-1 rounded-sm font-mono text-sm">${text}</code>`;
  }

  // Table renderer
  renderer.table = function (header, body) {
    return `
    <table class="border-collapse border border-gray-300 my-4 w-full">
      ${header}
      ${body}
    </table>
  `;
  }

  // Table row renderer
  renderer.tablerow = function (content) {
    return `<tr>${content}</tr>`;
  }

  // Table cell renderer
  renderer.tablecell = function (content, flags) {
    const tag = flags.header ? 'th' : 'td';
    const alignClass = flags.align ? `text-${flags.align}` : 'text-left';
    const baseClass = flags.header ? 'font-semibold p-2 border border-gray-300' : 'p-2 border border-gray-300';
    return `<${tag} class="${baseClass} ${alignClass}">${content}</${tag}>`;
  }

  // Code block renderer with Tailwind styling
  renderer.code = function (code, language) {
    const detectedLang = language && Prism.languages[language] ? language : 'plaintext'
    const highlightedCode = Prism.highlight(code, Prism.languages[detectedLang], detectedLang)
    return `
      <div class="bg-gray-800 rounded-md p-4">
        <div class="flex justify-between items-center mb-2">
          <span class="text-gray-400 text-sm">${language ?? ''}</span>
          <div class="flex items-center space-x-2 cursor-pointer" onclick="dkuAnswers_copyToClipboard(this)" data-clipboard-text="${encodeURIComponent(code)}">
            <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 12 12" fill="none" class="text-gray-400">
              <g opacity="0.7">
                <path
                  d="M5.23877 1.02441C4.41034 1.02441 3.73877 1.69599 3.73877 2.52441V7.5166C3.73877 8.34503 4.41034 9.0166 5.23877 9.0166H8.76416C9.59259 9.0166 10.2642 8.34503 10.2642 7.5166V2.52441C10.2642 1.69599 9.59259 1.02441 8.76416 1.02441H5.23877ZM4.73877 2.52441C4.73877 2.24827 4.96263 2.02441 5.23877 2.02441H8.76416C9.0403 2.02441 9.26416 2.24827 9.26416 2.52441V7.5166C9.26416 7.79274 9.0403 8.0166 8.76416 8.0166H5.23877C4.96263 8.0166 4.73877 7.79274 4.73877 7.5166V2.52441Z"
                  fill="currentColor"
                />
                <path
                  d="M1.73584 4.52867C1.73584 3.71648 2.12315 2.99474 2.72314 2.53809V8.51695H2.73639C2.75812 9.32642 3.42113 9.97594 4.23584 9.97594H4.4491C4.49024 9.98109 4.53217 9.98375 4.57471 9.98375H8.50943C8.05298 10.5865 7.32953 10.9759 6.51514 10.9759H4.23584C2.85513 10.9759 1.73584 9.85665 1.73584 8.47594V4.52867Z"
                  fill="currentColor"
                />
              </g>
            </svg>
            <span class="text-gray-400 text-sm">${translations['copy_code']}</span>
          </div>
        </div>
        <pre class="language-${detectedLang} overflow-auto"><code>${highlightedCode}</code></pre>
      </div>
    `
  }

  // Horizontal rule renderer
  renderer.hr = function () {
    return `<hr class="my-6 border-gray-300">`
  }
const strippedStr = str.replace(
  /(```[\s\S]*?```)|(`[^`\r\n]*`)|("(?:<([^>\s\/]+)([^>]*)>)")|<([^>\s\/]+)([^>]*)>/g,
  (
    fullMatch,
    fencedCode,  // ```…```
    inlineCode,  // `…`
    quotedWhole, // "<...>"
    quoteName,
    quoteRest,
    name,
    rest
  ) => {
    if (fencedCode || inlineCode) return fullMatch;

    // quoted "<...>" placeholder (used for PII)
    if (quotedWhole) {
      return `"&lt;${quoteName}${quoteRest}&gt;"`;
    }

    // unquoted "<...>"
    if (name && /^[a-z][a-z0-9-]*$/.test(name)) {
      return fullMatch; // potentially a real tag
    }
    return `&lt;${name}${rest}&gt;`;
  }
);

  // Patterns to extract citations
  const citationPattern = /<citation index="(\d+)"[\s\S]*?quote="([\s\S]*?)"><\/citation>/g
  const legacyCitationPattern =
    /\[source_index\](\d+)\[\/source_index\][\s\S]*?\[source_quote\]([\s\S]*?)\[\/source_quote\]/g
  let match
  const citations: any[] = []


  // Extract citations from markdown
  while ((match = citationPattern.exec(strippedStr)) !== null) {
    const index = match[1]
    const quote = match[2]?.replace(/\n/g, '<br />')
    citations.push({ index, quote })
  }
  while ((match = legacyCitationPattern.exec(strippedStr)) !== null) {
    const index = match[1]
    const quote = match[2]?.replace(/\n/g, '<br />')
    citations.push({ index, quote })
  }


  // Replace citations with placeholders
  const placeholderStr = strippedStr
    .replace(citationPattern, '<citation_placeholder>')
    .replace(legacyCitationPattern, '<citation_placeholder>')

  // Convert markdown to HTML
  const markdownContent = marked(placeholderStr, {
    renderer,
    headerIds: false,
    mangle: false
  })

  // Replace placeholders with Tailwind-styled citation popups
  let citIndex = 0
  const modifiedMarkdownContent = markdownContent.replace(/<citation_placeholder>/g, () => {
    const citation = citations[citIndex]
    const citationHtml = `
      <span class="text-[#5697f5] group relative inline-block cursor-pointer" id="query-${queryIndex}-citation-${citIndex}" data-index="${citation.index}"
        onclick="dkuAnswers_handleCitationClick(${citIndex},${citation.index}, ${queryIndex})">
        [${citation.index}]
        <span class="w-[200px] absolute left-0 mt-1 p-2 bg-white text-gray-700 rounded shadow-lg opacity-0 pointer-events-none transition-opacity duration-200 group-hover:opacity-100 z-10 text-sm">
          <b>“ </b><i>${citation.quote}</i><b> ”</b>
        </span>
      </span>
    `
    citIndex++
    return citationHtml
  })
  // Sanitize the final HTML
  return dompurify.sanitize(modifiedMarkdownContent, {
    ADD_ATTR: ['target', 'onclick', 'onmouseover', 'onmouseleave']
  })
}
