import { expect } from '@playwright/test';
import { test } from "../../../playwright.setup";

const loadPage = async (page, baseURL) => {
  expect(baseURL).toBeTruthy();
  await page.goto(baseURL!);
  await expect(
    page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
  ).not.toBeVisible();
  await expect(page.locator(".q-spinner")).toHaveCount(0);
};

// Helper to get Angular config scope
async function getConfigScope(page) {
  return page.evaluate(() => {
    try {
      const controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
      if (!controllerElement) return { error: 'Controller element not found' };
      
      const scope = angular.element(controllerElement).scope();
      return {
        config: scope.config,
        $apply: () => scope.$apply()
      };
    } catch (e) {
      return { error: e.message };
    }
  });
}

// Helper to set folder choices in directive scope
async function setFolderChoices(page, choices) {
  await page.evaluate((choices) => {
    // Find an element inside the directive's template
    const inputElement = document.querySelector('folder-selector[config-property="upload_folder"] input');
    if (!inputElement) return;
    
    // Get the scope from the input element (which is in the directive's isolated scope)
    const scope = angular.element(inputElement).scope();
    scope.folders = choices;
    scope.$apply();
  }, choices);
}

test('Folder Selector Directive - Selection, filtering and clearing', async ({ page, baseURL }) => {
  // Load and initialize page
  await loadPage(page, baseURL);
  await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible', timeout: 15000 });
  await page.waitForLoadState('networkidle');

  // Navigate to document upload section
  await page.click('a[href="#documentUpload"]');

  // Set mock folder choices directly in directive scope
  const mockChoices = [
    { label: 'Project Folder', value: 'project_folder' },
    { label: 'Dataiku Resources', value: 'dataiku_resources' },
    { label: 'User Uploads', value: 'user_uploads' },
    { label: 'Dataiku Analytics', value: 'dataiku_analytics' }
  ];
  await setFolderChoices(page, mockChoices);

  // Locate the folder selector
  const folderSelector = page.locator('folder-selector[config-property="upload_folder"]');
  
  // ===== Test 1: Select first option =====
  await test.step('Select first folder option', async () => {
    // Open dropdown
    await folderSelector.locator('.dku-icon-chevron-down-16').click();
    await page.waitForSelector('div[ng-if="isFolderDropdownOpen"]', { state: 'visible' });
    
    // Select first option
    await folderSelector.locator('div[ng-repeat*="folder in folders"]').first().click();
    
    // Verify config update
    const scope = await getConfigScope(page);
    expect(scope.config.upload_folder).toBe('project_folder');
    
    // Verify UI shows selected value
    const inputValue = await folderSelector.locator('input').inputValue();
    expect(inputValue).toBe('Project Folder');
  });

  // ===== Test 2: Filter and select =====
  await test.step('Filter and select folder', async () => {
    // Clear current selection
    await folderSelector.locator('.dku-icon-dismiss-16').click();
    
    // Type filter text
    await folderSelector.locator('input').fill('dataiku');
    
    // Wait for filtered results
    await page.waitForTimeout(500);
    
    // Verify filtered options
    const filteredOptions = await folderSelector.locator('div[ng-repeat*="folder in folders"]').all();
    expect(filteredOptions.length).toBe(2);
    
    // Select first filtered option
    await filteredOptions[0].click();
    
    // Verify config update
    const scope = await getConfigScope(page);
    expect(scope.config.upload_folder).toBe('dataiku_resources');
    
    // Verify UI shows selected value
    const inputValue = await folderSelector.locator('input').inputValue();
    expect(inputValue).toBe('Dataiku Resources');
  });

  // ===== Test 3: Clear selection =====
  await test.step('Clear folder selection', async () => {
    // Click clear button
    await folderSelector.locator('.dku-icon-dismiss-16').click();
    
    // Verify config update
    const scope = await getConfigScope(page);
    expect(scope.config.upload_folder).toBeNull();
    
    // Verify UI shows empty input
    const inputValue = await folderSelector.locator('input').inputValue();
    expect(inputValue).toBe('');
    
    // Verify dropdown shows all options when opened
    await folderSelector.locator('.dku-icon-chevron-down-16').click();
    await page.waitForSelector('div[ng-if="isFolderDropdownOpen"]', { state: 'visible' });
    const allOptions = await folderSelector.locator('div[ng-repeat*="folder in folders"]').all();
    expect(allOptions.length).toBe(4);
  });
});