import { expect } from '@playwright/test';
import { test } from "../../../playwright.setup";

// Helper function to verify LLM selection state
async function verifyLlmSelection(page, expectedDisplayName = null) {
  const selectionState = await page.evaluate(() => {
    try {
      const element = document.querySelector('input[ng-model="llmFilter"]'); 
      if (!element) return { error: 'LLM filter input not found' };

      const scope = angular.element(element).scope();
      const controllerScope = angular.element(document.querySelector('[ng-controller="LLMConfigController"]')).scope();

      const configProp = scope.configProperty || 'llm_id';

      return {
        isDropdownOpen: scope.isLLMDropdownOpen,
        configProperty: configProp,
        configValue: controllerScope.config[configProp],
        selectedLLM: scope.selectedLLM,
        llmFilter: scope.llmFilter,
        isValid: controllerScope.config[configProp] === (scope.selectedLLM ? scope.selectedLLM.llmId : null),
        availableLLMs: scope.llms.length,
        llmsByType: Object.keys(scope.llmsByType || {})
      };
    } catch (e) {
      return { error: e.message, stack: e.stack };
    }
  });

  console.log('LLM Selection State:', JSON.stringify(selectionState, null, 2));

  if (selectionState.error) {
    throw new Error(`Failed to get LLM selection state: ${selectionState.error}`);
  }

  return selectionState;
}

const loadPage = async (page, baseURL) => {
  expect(baseURL).toBeTruthy();
  await page.goto(baseURL!);
  await expect(
    page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
  ).not.toBeVisible();
  await expect(page.locator(".q-spinner")).toHaveCount(0);
};

test('LLM Configuration Form - Complete LLM selector functionality', async ({ page, baseURL }) => {
  // Navigate and setup
  await loadPage(page, baseURL);
  await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible' });

  // Navigate to LLM section
  await page.click('a[href="#llmConfig"]');
  await page.waitForSelector('#section-llmConfig', { state: 'visible' });

  // Target the LLM selector specifically
  const llmSelector = page.locator('unique-llm-selector').first();

  console.log('=== Step 1: Click dropdown and verify it opens ===');
  
  // Click on the dropdown arrow to open
  await llmSelector.locator('.dku-icon-chevron-down-16').click({ timeout: 5000 });

  // Wait for dropdown to be fully loaded
  await page.waitForSelector('div[ng-if="isLLMDropdownOpen"]', {
    state: 'visible',
    timeout: 10000
  });

  // Wait for the LLM options to be loaded
  await page.waitForSelector('div[ng-repeat*="llm in llmsInType"]', {
    state: 'visible',
    timeout: 10000
  });

  // Verify dropdown is open
  let state = await verifyLlmSelection(page);
  expect(state.isDropdownOpen).toBe(true);
  expect(state.availableLLMs).toBeGreaterThan(0);
  
  console.log('✅ Dropdown opened successfully');
  console.log(`Available LLMs: ${state.availableLLMs}`);

  console.log('=== Step 2: Select first LLM ===');
  
  // Click the first LLM option (excluding headers)
  await page.locator('div[ng-repeat*="llm in llmsInType"][ng-if*="llmMatchesFilter"]')
    .first()
    .click({ timeout: 10000 });

  // Wait for selection to process
  await page.waitForTimeout(1000);

  // Verify something is selected
  state = await verifyLlmSelection(page);
  expect(state.selectedLLM).toBeTruthy();
  expect(state.llmFilter).toBeTruthy();
  expect(state.isValid).toBe(true);
  
  console.log('✅ LLM selected:', state.llmFilter);
  console.log('Selected LLM ID:', state.selectedLLM.llmId);

  console.log('=== Step 3: Test clear button ===');
  
  // Click the clear button within the LLM selector
  await llmSelector.locator('.dku-icon-dismiss-16').click({ timeout: 5000 });


  // Wait for clearing to process
  await page.waitForTimeout(1000);

  // Verify selection is cleared
  state = await verifyLlmSelection(page);
  expect(state.selectedLLM).toBeNull();
  expect(state.llmFilter).toBe('');
  expect(state.configValue).toBeNull();
  expect(state.isValid).toBe(true); // Should be valid when null (depending on required field)
  
  console.log('✅ Clear button worked successfully');

  console.log('=== Step 4: Re-select LLM and verify config ===');
  
  // The dropdown should reopen after clearing, if not open it
  const isDropdownOpen = await page.isVisible('div[ng-if="isLLMDropdownOpen"]');
  if (!isDropdownOpen) {
    await llmSelector.locator('.dku-icon-chevron-down-16').click();
    await page.waitForSelector('div[ng-if="isLLMDropdownOpen"]', { state: 'visible' });
  }

  // Wait for LLM options to be available again
  await page.waitForSelector('div[ng-repeat*="llm in llmsInType"][ng-if*="llmMatchesFilter"]', {
    timeout: 10000
  });

  // Select the first LLM again
  await page.locator('div[ng-repeat*="llm in llmsInType"][ng-if*="llmMatchesFilter"]')
    .first()
    .click({ timeout: 10000 });

  // Click outside to trigger any final blur events
  await page.waitForTimeout(1000);

  // Final verification
  state = await verifyLlmSelection(page);
  
  console.log('Final LLM selection state:', {
    configValue: state.configValue,
    selectedLLMId: state.selectedLLM?.llmId,
    llmFilter: state.llmFilter,
    isValid: state.isValid,
    isDropdownOpen: state.isDropdownOpen
  });

  // Main assertions - verify that config.llm_id equals selectedLLM.llmId
  expect(state.selectedLLM).toBeTruthy();
  expect(state.selectedLLM.llmId).toBeTruthy();
  expect(state.configValue).toBe(state.selectedLLM.llmId);
  expect(state.llmFilter).toBe(state.selectedLLM.llmDisplayName);
  expect(state.isValid).toBe(true);
  expect(state.isDropdownOpen).toBe(false); // Should close after selection

  console.log('✅ LLM selection and config verification successful');
  console.log(`Final selected LLM: ${state.selectedLLM.llmDisplayName} (ID: ${state.selectedLLM.llmId})`);
  
  // Additional verification using the input value
  const selectedInputValue = await page.inputValue('input[ng-model="llmFilter"]');
  expect(selectedInputValue).toBeTruthy();
  expect(selectedInputValue).toBe(state.selectedLLM.llmDisplayName);
  
  console.log('✅ Input field verification passed');
});
