import copy
from typing import Optional

from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger
from dataikuapi.dss.llm import DSSLLMCompletionQuery

FALLBACK_LLM_KEY = "ENABLE_FALLBACK_LLM"


def get_fallback_id() -> Optional[str]:
    webapp_config = dataiku_api.webapp_config
    advanced_hidden_settings = webapp_config.get("advanced_hidden_settings")
    if not advanced_hidden_settings:
        return None
    id_list = [
        setting_item.get("to") for setting_item in advanced_hidden_settings if str(setting_item.get("from")) == FALLBACK_LLM_KEY
    ]
    return id_list[0] if id_list else None


def is_fallback_enabled(completion: DSSLLMCompletionQuery) -> bool:
    fallback_id = get_fallback_id()
    if fallback_id is None:
        return False
    current_id = completion.llm_id
    if current_id == fallback_id:
        logger.error(f"Current LLM id: {current_id} is the same as fallback id: {fallback_id}")
        return False
    return True


def get_fallback_completion(completion: DSSLLMCompletionQuery) -> Optional[DSSLLMCompletionQuery]:
    current_completion = copy.deepcopy(completion)
    fallback_llm_id = get_fallback_id()
    fallback_completion: DSSLLMCompletionQuery = dataiku_api.default_project.get_llm(fallback_llm_id).new_completion()
    for k, v in current_completion.settings.items():
        fallback_completion.settings[k] = v
    if len(current_completion.cq["messages"]) > 0:
        fallback_completion.cq["messages"] = current_completion.cq["messages"]
    logger.debug(f"Using fallback LLM with ID: {fallback_completion.llm.llm_id}.")
    return fallback_completion
