import { io, Socket } from 'socket.io-client'
export interface WebSocketEventHandlers {
  onMessage?: (data: any) => void
  onLogQueryComplete?: (data: any) => void
  onError?: (error: any) => void
  onDisconnect?: () => void
}
export class WebSocketApi {
  private static socket: Socket | undefined
  private static _host: string | undefined
  private static path: string | undefined

  public static init(host: string, path: string) {
    this._host = host
    this.path = path + 'stream'
  }

  public static setupHandlers(handlers: WebSocketEventHandlers = {}): void {
    if(!this._host) return
    this.socket = io(this._host, {
      path: this.path,
      transports: ['websocket', 'polling']
    })
    // Set up event handlers
    if (handlers.onMessage) {
      this.socket.on('message', handlers.onMessage)
    }
    if (handlers.onLogQueryComplete) {
      this.socket.on('log_query_complete', handlers.onLogQueryComplete)
    }

    if (handlers.onError) {
      this.socket.on('error', handlers.onError)
    }

    if (handlers.onDisconnect) {
      this.socket.on('disconnect', handlers.onDisconnect)
    }
  }

  public static sendMessage(message: any) {
    if(!this.checkAndReconnect() || !this.socket) return
    console.debug('WebSocket is open now.', JSON.stringify(message))
    this.socket.emit('message', JSON.stringify(message))
  }
  public static sendLogQuery(message: any) {
    if(!this.checkAndReconnect()) return
    console.debug('log query', JSON.stringify(message))
    this.socket?.emit('log_query', JSON.stringify(message))
  }
  public static close(): void {
    if (!this.socket) {
      console.warn('WebSocket is not initialized.')
      return
    }
    this.socket.close()
  }
  private static checkAndReconnect() {
    if (!this.socket) {
      console.error('WebSocket is not open. Message not sent.', this.socket)
      return false
    }
    if (this.socket.disconnected) {
      this.socket.connect()
    }
    return true
  }
}
