import { expect } from '@playwright/test';
import { test } from "../../../playwright.setup";

const loadPage = async (page, baseURL) => {
    expect(baseURL).toBeTruthy();
    await page.goto(baseURL!);
    await expect(
        page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
    ).not.toBeVisible();
    await expect(page.locator(".q-spinner")).toHaveCount(0);
};

// Helper to get Angular config scope
async function getConfigScope(page) {
    return page.evaluate(() => {
        try {
            const controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
            if (!controllerElement) return { error: 'Controller element not found' };

            const scope = angular.element(controllerElement).scope();
            return {
                config: scope.config,
                $apply: () => scope.$apply()
            };
        } catch (e) {
            return { error: e.message };
        }
    });
}

// Helper to set folders in folderMultiSelector scope
async function setMultiSelectorFolders(page, configProperty, mockData) {
    await page.evaluate(({ configProperty, mockData }) => {
        // Find the directive element
        const selectorElement = document.querySelector(`folder-multi-selector[config-property="${configProperty}"]`);
        if (!selectorElement) return;

        // Access the scope through a descendant element
        const inputElement = selectorElement.querySelector('div[ng-click="toggleDropdown($event)"]');
        const scope = angular.element(inputElement).scope();

        // Process mock data to match the directive's format
        const normalizedData = {};
        mockData.forEach((folder, i) => {
            const type = folder.type || "General";
            if (!normalizedData[type]) normalizedData[type] = [];
            normalizedData[type].push({
                value: folder.value,
                label: folder.label,
                type: type
            });
        });

        // Set directive's internal state
        scope.foldersByType = normalizedData;
        scope.folders = Object.values(normalizedData).flat();

        scope.$apply();
    }, { configProperty, mockData });
}



test('Folder Multi Selector - Selection and Clearing', async ({ page, baseURL }) => {
    // Load and initialize page
    await loadPage(page, baseURL);
    await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible', timeout: 15000 });
    await page.waitForLoadState('networkidle');

    // Navigate to user profile section
    await page.click('a[href="#retrievalMode"]');


    // Set retrieval method to KB via UI interaction
    await test.step('Set retrieval method to KB', async () => {
        // Locate the folder-selector for retrieval_mode
        const retrievalSelector = page.locator('folder-selector[config-property="retrieval_mode"]');

        // Open dropdown
        await retrievalSelector.locator('.dku-icon-chevron-down-16').click();
        await page.waitForSelector('div[ng-if="isFolderDropdownOpen"]', { state: 'visible' });

        // Select "Use knowledge bank retrieval" option
        const kbOption = retrievalSelector.locator('div[ng-repeat*="folder in folders"]', {
            hasText: 'Use knowledge bank retrieval (for searches within text)'
        }).first();

        await kbOption.click();

        // Verify selection was made
        await expect(retrievalSelector.locator('input')).toHaveValue('Use knowledge bank retrieval (for searches within text) ');
    });

    // Define mock folder data
    const mockFolders = [
        { value: 'category1', label: 'Category 1', type: 'Type A' },
        { value: 'category2', label: 'Category 2', type: 'Type A' },
        { value: 'category3', label: 'Category 3', type: 'Type B' },
        { value: 'category4', label: 'Category 4', type: 'Type B' }
    ];

    const configProperty = 'knowledge_sources_context_metadata';
    await setMultiSelectorFolders(page, configProperty, mockFolders);

    // Locate the multi-selector
    const multiSelector = page.locator(`folder-multi-selector[config-property="${configProperty}"]`);

    // ===== Test 1: Select one element =====
    await test.step('Select single folder', async () => {
        // Open dropdown
        await multiSelector.locator('div[ng-click="toggleDropdown($event)"]').click();
        await page.waitForSelector('[ng-if="isDropdownOpen"]', { state: 'visible' });

        // Check the first checkbox
        await multiSelector.locator('input[type="checkbox"]').first().check();

        // Close dropdown
        await page.click('body'); // Click elsewhere to close

        // Verify config update
        const scope = await getConfigScope(page);
        expect(scope.config[configProperty]).toEqual(['category1']);
    });

    // ===== Test 2: Select two elements =====
    await test.step('Select multiple folders', async () => {
        // Open dropdown
        await multiSelector.locator('div[ng-click="toggleDropdown($event)"]').click();
        await page.waitForSelector('[ng-if="isDropdownOpen"]', { state: 'visible' });

        // Get all checkboxes and check first and third
        const checkboxes = await multiSelector.locator('input[type="checkbox"]').all();
        await checkboxes[0].check(); // First checkbox
        await checkboxes[2].check(); // Third checkbox

        // Close dropdown
        await page.click('body'); // Click elsewhere to close

        // Verify config update
        const scope = await getConfigScope(page);
        expect(scope.config[configProperty]).toEqual(['category1', 'category3']);
    });

// ===== Test 3: Unselect a folder =====
await test.step('Unselect a folder', async () => {
    // Based on your HTML template, the selected items have different classes
    // They use "!bg-black !text-gray-200 !rounded-full" instead of "!bg-blue-100"
    
    // Find the chip for the first item - using the correct structure from your HTML
    const firstChip = multiSelector.locator('div[ng-repeat="item in getVisibleItems()"]', {
        hasText: 'Category 1'
    }).first();

    // Click the remove button in the chip
    await firstChip.locator('button[ng-click="removeItem(item, $event)"]').click();

    // Verify config update
    const scope = await getConfigScope(page);
    expect(scope.config[configProperty]).toEqual(['category3']);
});

// ===== Test 4: Clear all selections =====
await test.step('Clear all selections', async () => {
    // Get all chips with the correct class from your HTML
    const chips = multiSelector.locator('div[ng-repeat="item in getVisibleItems()"]');
    
    // Remove each chip
    const count = await chips.count();
    for (let i = 0; i < count; i++) {
        // Always remove first chip since list updates
        await chips.first().locator('button').click();
        await page.waitForTimeout(200); // Small delay for UI update
    }

    // Verify config update
    const scope = await getConfigScope(page);
    expect(scope.config[configProperty]).toEqual([]);
});});