import { expect } from '@playwright/test';
import { test } from "../../../playwright.setup";

const loadPage = async (page, baseURL) => {
  expect(baseURL).toBeTruthy();
  await page.goto(baseURL!);
  await expect(
    page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
  ).not.toBeVisible();
  await expect(page.locator(".q-spinner")).toHaveCount(0);
};

// Helper to get Angular config scope
async function getConfigScope(page) {
  return page.evaluate(() => {
    try {
      const controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
      if (!controllerElement) return { error: 'Controller element not found' };

      const scope = angular.element(controllerElement).scope();
      return {
        config: scope.config,
        $apply: () => scope.$apply()
      };
    } catch (e) {
      return { error: e.message };
    }
  });
}

test('Language Mappings Directive - Initial state, deletion and addition', async ({ page, baseURL }) => {
  // Load and initialize page
  await loadPage(page, baseURL);
  await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible', timeout: 15000 });
  await page.waitForLoadState('networkidle');

  // Navigate to user profile section
  await page.click('a[href="#userProfileSettings"]');

  // Get initial config state
  const initialScope = await getConfigScope(page);
  const initialLanguages = initialScope.config.user_profile_languages;
  const initialCount = initialLanguages.length;

  // Locate the languages mapping section
  const languagesSection = page.locator('language-mappings:has-text("Languages available to user")');

  // ===== Test 1: Verify initial values =====
  await test.step('Verify initial language mappings', async () => {
    // Get all mapping items
    const mappingItems = languagesSection.locator('.language-mapping-item');
    await expect(mappingItems).toHaveCount(initialCount);

    // Verify each item's content
    for (let i = 0; i < initialCount; i++) {
      const nameInput = mappingItems.nth(i).locator('input').first();
      const codeInput = mappingItems.nth(i).locator('input').nth(1);

      const codeValue = await codeInput.inputValue();
      const nameValue = await nameInput.inputValue();

      // Find matching language in config
      const matchingLang = initialLanguages.find(lang =>
        lang.from === codeValue && lang.to === nameValue
      );

      expect(matchingLang).toBeTruthy();
    }
  });

  // ===== Test 2: Delete an element =====
  await test.step('Delete a language mapping', async () => {
    // Get first mapping item
    const firstItem = languagesSection.locator('.language-mapping-item').first();
    const firstCode = await firstItem.locator('input').first().inputValue();
    const firstName = await firstItem.locator('input').nth(1).inputValue();

    // Delete the item
    await firstItem.locator('button').click();

    // Verify UI update
    const mappingItems = languagesSection.locator('.language-mapping-item');
    await expect(mappingItems).toHaveCount(initialCount - 1);

    // Verify config update
    const updatedScope = await getConfigScope(page);
    const updatedLanguages = updatedScope.config.user_profile_languages;

    // Verify deleted item is gone
    const deletedItemExists = updatedLanguages.some(lang =>
      lang.from === firstCode && lang.to === firstName
    );
    expect(deletedItemExists).toBe(false);

    // Verify count decreased
    expect(updatedLanguages.length).toBe(initialCount - 1);
  });

  // ===== Test 3: Add a new element =====
  await test.step('Add a new language mapping', async () => {
    // Click add button
    await languagesSection.locator('button:has-text("Add a language")').click();

    // Get the new item (last in the list)
    const mappingItems = languagesSection.locator('.language-mapping-item');
    const newItem = mappingItems.last();

    // Fill in new values
    const newCode = 'nl';
    const newName = 'Dutch';

    await newItem.locator('input').first().fill(newName);
    await newItem.locator('input').nth(1).fill(newCode);

    // Verify config update
    const updatedScope = await getConfigScope(page);
    const updatedLanguages = updatedScope.config.user_profile_languages;

    // Verify new item exists
    const newItemExists = updatedLanguages.some(lang =>
      lang.from === newCode && lang.to === newName
    );
    expect(newItemExists).toBe(true);

    // Verify count increased
    expect(updatedLanguages.length).toBe(initialCount); // Back to original count
  });
});
