import { expect } from '@playwright/test';
import { test } from "../../../playwright.setup";

const loadPage = async (page, baseURL) => {
  expect(baseURL).toBeTruthy();
  await page.goto(baseURL!);
  await expect(
    page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
  ).not.toBeVisible();
  await expect(page.locator(".q-spinner")).toHaveCount(0);
};

// Helper to get Angular config scope
async function getConfigScope(page) {
  return page.evaluate(() => {
    try {
      const controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
      if (!controllerElement) return { error: 'Controller element not found' };
      
      const scope = angular.element(controllerElement).scope();
      return {
        config: scope.config,
        $apply: () => scope.$apply()
      };
    } catch (e) {
      return { error: e.message };
    }
  });
}

test('Feedback Choices Directive - Add and verify choices', async ({ page, baseURL }) => {
  // Load and initialize page
  await loadPage(page, baseURL);
  await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible', timeout: 15000 });
  await page.waitForLoadState('networkidle');

  // Navigate to conversation store section
  await page.click('a[href="#conversationStore"]');
//   await page.waitForSelector('#section-conversationStoreConfig', { state: 'visible' });

  // Reset feedback arrays
  const scope = await getConfigScope(page);
  if (scope.error) throw new Error(scope.error);
  await page.evaluate(() => {
    const controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
    const scope = angular.element(controllerElement).scope();
    scope.config.feedback_positive_choices = [];
    scope.config.feedback_negative_choices = [];
    scope.$apply();
  });

  // ===== Test Positive Feedback =====
  const positiveSection = page.locator('label:has-text("Positive feedback choices") + feedback-choices');
  await test.step('Add positive feedback choices', async () => {
    const choices = ['Good', 'Great', 'Amazing'];
    
    for (const choice of choices) {
      await positiveSection.locator('button:has-text("Add an element")').click();
      const inputs = positiveSection.locator('.feedback-item input');
      const lastInput = inputs.last();
      await lastInput.fill(choice);
    }
  });

  // ===== Test Negative Feedback =====
  const negativeSection = page.locator('label:has-text("Negative feedback choices") + feedback-choices');
  await test.step('Add negative feedback choices', async () => {
    const choices = ['Terrible', 'Bad', 'Not Good'];
    
    for (const choice of choices) {
      await negativeSection.locator('button:has-text("Add an element")').click();
      const inputs = negativeSection.locator('.feedback-item input');
      const lastInput = inputs.last();
      await lastInput.fill(choice);
    }
  });

  // ===== Verify Config Updates =====
  await test.step('Verify config object updates', async () => {
    const updatedScope = await getConfigScope(page);
    if (updatedScope.error) throw new Error(updatedScope.error);
    
    // Verify positive choices
    expect(updatedScope.config.feedback_positive_choices).toEqual([
      'Good', 'Great', 'Amazing'
    ]);
    
    // Verify negative choices
    expect(updatedScope.config.feedback_negative_choices).toEqual([
      'Terrible', 'Bad', 'Not Good'
    ]);
    
    // Verify UI matches config
    const positiveInputs = await positiveSection.locator('input').all();
    for (let i = 0; i < positiveInputs.length; i++) {
      expect(await positiveInputs[i].inputValue()).toBe(
        updatedScope.config.feedback_positive_choices[i]
      );
    }
    
    const negativeInputs = await negativeSection.locator('input').all();
    for (let i = 0; i < negativeInputs.length; i++) {
      expect(await negativeInputs[i].inputValue()).toBe(
        updatedScope.config.feedback_negative_choices[i]
      );
    }
  });

  // ===== Test Item Removal =====
  await test.step('Test item removal', async () => {
    // Remove first positive item
    await positiveSection.locator('.feedback-item').first().locator('button').click();
    
    // Verify config update
    const updatedScope = await getConfigScope(page);
    expect(updatedScope.config.feedback_positive_choices).toEqual([
      'Great', 'Amazing'
    ]);
    
    // Verify UI update
    expect(await positiveSection.locator('.feedback-item').count()).toBe(2);
  });
});