import { expect } from '@playwright/test';
import { test } from "../../../playwright.setup";
import fs from 'fs';
import path from 'path';


const loadPage = async (page, baseURL) => {
    expect(baseURL).toBeTruthy();
    await page.goto(baseURL!);
    await expect(
        page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
    ).not.toBeVisible();
    await expect(page.locator(".q-spinner")).toHaveCount(0);
};

async function getConfigScope(page) {
    return page.evaluate(() => {
        try {
            const controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
            if (!controllerElement) return { error: 'Controller element not found' };

            const scope = angular.element(controllerElement).scope();
            return {
                config: scope.config,
                $apply: () => scope.$apply()
            };
        } catch (e) {
            return { error: e.message };
        }
    });
}

test('File Upload Directive - Basic Operations', async ({ page, baseURL }) => {
    await loadPage(page, baseURL);
    await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible', timeout: 15000 });
    await page.waitForLoadState('networkidle');

    // Navigate to Interface Customization section
    await page.click('a[href="#endUserConfiguration"]');
    await page.waitForSelector('custom-file-uploader', { state: 'visible' });

    // Global CSS file uploader setup
    const cssUploader = page.locator('custom-file-uploader[expected-file-name="custom.css"]');
    const fileUploader = page.locator('custom-file-uploader[config-property="custom_logo_file_name"]');

    // Test 1: Upload valid image file
    await test.step('Upload valid logo image', async () => {

        const fileInput = fileUploader.locator('input[type="file"]');
        await fileInput.setInputFiles('./tests/playwright/custom-ui-settings/test-fixtures/test-logo.png');

        await expect(fileUploader.locator('text="Upload successful!"')).toBeVisible();
        await expect(fileUploader.locator('text="custom_logo_file_name.png"')).toBeVisible();

        const scope = await getConfigScope(page);
        expect(scope.config.custom_logo_file_name).toBe('custom_logo_file_name.png');
    });

    // Test 2: Upload invalid file type
    await test.step('Reject invalid file type', async () => {
        const errorDiv = page.locator('div[ng-if="clientErrorMessage"]')


        await fileUploader.locator('input[type="file"]').setInputFiles('./tests/playwright/custom-ui-settings/test-fixtures/invalid.pdf');

        // Wait for error div visibility
        await errorDiv.waitFor({ state: 'visible', timeout: 5000 });

        // Verify exact error message
        await expect(errorDiv).toContainText('Only image files are accepted', {
            useInnerText: true,
            timeout: 3000
        });

        // Also verify red border styling
        const uploadArea = fileUploader.locator('.drag-drop-area');
        await expect(uploadArea).toHaveClass(/!border-red-500/);

    });

    // Test 3: Remove uploaded file
    await test.step('Remove uploaded logo', async () => {
        page.once('dialog', dialog => dialog.accept());
        // Invalid file name
        await fileUploader.locator('button >> i.dku-icon-trash-16').click();
        await expect(fileUploader.locator('text="custom_logo_file_name.png"')).toBeHidden();
        const scope = await getConfigScope(page);
        expect(scope.config.custom_logo_file_name).toBeNull();
    });

    // Test 4: CSS file validation
    // Test 4: CSS file validation
    await test.step('Validate CSS file name', async () => {
        // Find the error div that's a sibling or child of the cssUploader
        const errorDiv = cssUploader.locator('div[ng-if="clientErrorMessage"]');
        // OR if it's a preceding sibling:
        // const errorDiv = cssUploader.locator('xpath=./preceding-sibling::div[@ng-if="clientErrorMessage"][1]');

        // Correct file name
        await cssUploader.locator('input[type="file"]').setInputFiles('./tests/playwright/custom-ui-settings/test-fixtures/custom.css');
        await expect(cssUploader.locator('text="Upload successful!"')).toBeVisible();

        // Invalid file name
        await cssUploader.locator('input[type="file"]').setInputFiles('./tests/playwright/custom-ui-settings/test-fixtures/wrong-name.css');

        await expect(errorDiv).toContainText('Invalid file name: wrong-name.css', {
            useInnerText: true,
            timeout: 3000
        });
    });
    // Test 5: Drag and drop functionality
    await test.step('Handle drag and drop', async () => {
        await fileUploader.locator('.drag-drop-area').hover();
        await page.dispatchEvent('.drag-drop-area', 'dragover');

        const dataTransfer = await page.evaluateHandle(() => {
            const dt = new DataTransfer();
            const file = new File(['logo'], 'test-drag.png', { type: 'image/png' });
            dt.items.add(file);
            return dt;
        });

        await page.dispatchEvent('.drag-drop-area', 'drop', { dataTransfer });
        await expect(fileUploader.locator('text="custom_logo_file_name.png"')).toBeVisible({ timeout: 10000 });
    });
});

test('File Upload Directive - Theme Folder Creation', async ({ page, baseURL }) => {
    await loadPage(page, baseURL);
    await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible', timeout: 15000 });
    await page.waitForLoadState('networkidle');

    // Navigate to Interface Customization section
    await page.click('a[href="#endUserConfiguration"]');

    // Test theme folder auto-creation on first upload
    await test.step('Create theme folder automatically', async () => {
        const initialThemeName = 'Default_customization';
        const uploader = page.locator('custom-file-uploader[config-property="custom_logo_file_name"]');

        // Verify initial state
        let scope = await getConfigScope(page);
        expect(scope.config.custom_theme_name).toBe(initialThemeName);

        // Trigger new upload
        await uploader.locator('input[type="file"]').setInputFiles('./tests/playwright/custom-ui-settings/test-fixtures/new-theme-logo.png');

        // Verify new theme folder was created
        // Small delay to ensure Angular has processed everything
        await page.waitForTimeout(1000);

        scope = await getConfigScope(page);
        expect(scope.config.custom_theme_name).not.toBe(initialThemeName);
    });
});