import { expect } from '@playwright/test';
import { test } from "../../../../playwright.setup";

const loadPage = async (page, baseURL) => {
    expect(baseURL).toBeTruthy();
    await page.goto(baseURL!);
    await expect(
        page.locator("div").filter({ hasText: "Loading data..." }).nth(2)
    ).not.toBeVisible();
    await expect(page.locator(".q-spinner")).toHaveCount(0);
};

// Helper to get Angular config scope
async function getConfigScope(page) {
    return page.evaluate(() => {
        try {
            const controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
            if (!controllerElement) return { error: 'Controller element not found' };

            const scope = angular.element(controllerElement).scope();
            return {
                config: scope.config,
                invalidSections: scope.invalidSections,
                $apply: () => scope.$apply()
            };
        } catch (e) {
            return { error: e.message };
        }
    });
}

test('Form Validation - LLM and Datasets Section', async ({ page, baseURL }) => {
    // Load and initialize page
    await loadPage(page, baseURL);
    await page.waitForSelector('[ng-controller="LLMConfigController"]', { state: 'visible' });

    // Navigate to LLM and Datasets section
    await page.locator('a[href="#llmConfig"]').click();
    await page.waitForSelector('#section-llmConfig', { state: 'visible' });

    // Save initial valid state
    const initialScope = await getConfigScope(page);
    const validLLMId = initialScope.config.llm_id;
    const validLoggingDataset = initialScope.config.logging_dataset;
    const validProfileDataset = initialScope.config.user_profile_dataset;

    // ===== Test validation =====
    // Clear required fields
    await test.step('Clear required fields', async () => {
        // Clear LLM selector
        await page.locator('unique-llm-selector[config-property="llm_id"]')
            .locator('.dku-icon-dismiss-16')
            .click();

        // Close dropdown
        await page.locator("body").click({ position: { x: 0, y: 0 } });


        // Clear logging dataset selector
        const loggingDataset = page.locator('dataset-selector[config-property="logging_dataset"]');
        if (await loggingDataset.locator('.dku-icon-dismiss-16').count() > 0) {
            await loggingDataset.locator('.dku-icon-dismiss-16').click();
        }

        // Close dropdown
        await page.locator("body").click({ position: { x: 0, y: 0 } });


        // Clear logging dataset selector
        const userProfileDataset = page.locator('dataset-selector[config-property="user_profile_dataset"]');
        if (await userProfileDataset.locator('.dku-icon-dismiss-16').count() > 0) {
            await userProfileDataset.locator('.dku-icon-dismiss-16').click();
        }

        // Close dropdown
        await page.locator("body").click({ position: { x: 0, y: 0 } });


    });

    // Attempt to save
    await test.step('Attempt to save invalid form', async () => {
        // Click the Save and view webapp button
        await page.locator('a.btn.btn--primary:has-text("Save and view webapp")').click();

        // Wait for error message to appear
        await page.waitForSelector('#globalErrorBanner', { state: 'visible' });
    });

    // ===== Verify validation =====
    await test.step('Verify validation errors', async () => {
        // Verify error banner
        const banner = page.locator('#globalErrorBanner');
        await expect(banner).toBeVisible();
        await expect(banner).toContainText('Some information is missing or incorrect');
        await expect(banner).toContainText('LLM and Datasets');


        // Check if dataset selector has red border
        const datasetHasError = await page.evaluate(() => {
            const input = document.querySelector('input[ng-model="datasetFilter"]');
            console.log('INPUT ELEMENT', input)
            if (!input) return false;
            
            const container = input.closest('div');
            console.log(container,container?.classList.contains('border-red-500'),container?.classList.contains('ring-red-500'))
            return container && 
                   container.classList.contains('!border-red-500') && 
                   container.classList.contains('!ring-red-500');
        });
        expect(datasetHasError).toBe(true);


        // Check if dataset selector has red border
        const profileHasError = await page.evaluate(() => {
            const input = document.querySelectorAll('input[ng-model="datasetFilter"]')[1]
            console.log('INPUT ELEMENT', input)
            if (!input) return false;
            
            const container = input.closest('div');
            console.log(container,container?.classList.contains('border-red-500'),container?.classList.contains('ring-red-500'))
            return container && 
                   container.classList.contains('!border-red-500') && 
                   container.classList.contains('!ring-red-500');
        });
        expect(profileHasError).toBe(true);


        // Verify invalidSections array
        const scope = await getConfigScope(page);
        expect(scope.invalidSections).toEqual([
            expect.objectContaining({ id: 'llmConfig', displayName: 'LLM and Datasets' })
        ]);
    });

});
