import json
import locale
import os
from datetime import datetime
from logging.config import dictConfig
from pathlib import Path

from common.backend.utils.dataiku_api import dataiku_api

year, date = datetime.now().strftime("%Y"), datetime.now().strftime("%Y-%m-%d")
# Replace by your default project key that you are working on in dev
DEFAULT_PROJECT_KEY = "NEW_FEATURE_SQL_RET"
LOGGING_DATASET = "chat_logs"
FEEDBACK_DATASET = "general_feedbacks"
KNOWLEDGE_BANK_ID = "W0Caz5vs"  # "8v41yY7i", "None"
UPLOAD_OLDER_ID = "nJn6aY7q"

# TODO : Add dip_home to a .env file
DEFAULT_CONFIG = {
    # put your webapp desired config
    "webapp_config": {
        "primer_prompt": f"You are a general purpose assistant, be as concise as possible, The current Year is {year}. The current Date is {date} ",
        "system_prompt": f"""Please act as an assistant and comply to the following principles:
        Understand and Clarify the Task: Begin by comprehensively understanding the user's request. If necessary, ask clarifying questions to ensure accuracy in the response.""",
        "memory_token_limit": 8000,
        "web_app_title": "Ask directly questions to your document corpus",
        "web_app_subheading": "Answers are sourced from the processed documents, delivered in natural language and accompanied by the most relevant references.",
        "example_questions": ["What are the primary causes of the recent increase in global temperatures?", "How does quantum computing differ from classical computing in terms of processing information?", "What are the most used languages"],
        "language": "en",
        "llm_configuration": {
            "mode": "PRESET",
            "name": "Answer GPT 35"
        },
        "logging_dataset": LOGGING_DATASET,
        "feedback_positive_choices": [],
        "feedback_negative_choices": [],
        "general_feedback_dataset": FEEDBACK_DATASET,
        "retrieval_mode": "kb",
        "knowledge_retrieval_k": 20,
        "knowledge_retrieval_search_type": "similarity",
        "knowledge_retrieval_mmr_k": 20,
        "knowledge_retrieval_mmr_diversity": 0.25,
        "knowledge_retrieval_score_threshold": 0.8,
        "knowledge_sources_context_metadata": [],
        "web_app_input_placeholder": "Ask your question",
        "knowledge_bank_id": KNOWLEDGE_BANK_ID,
        "activate_knowledge_bank": True,
        "knowledge_bank_prompt": f"""You are Dataiku Chat, an assistant for Dataiku teams.
1. Your responses should be specific to the questions and respond professionally.
2. Avoid any Personally Identifiable Info like names in your responses. 
3. If you don't have the info say "You don't have it". Do not apologize for not having information.
4. Avoid preambles like "Based on the context provided" instead jump straight to the response.
5. You do not have access to the web/internet and you do not have information on real-time events, news, and weather.""",
        "llm_id": "openai:bs-openai:gpt-4o",
        "knowledge_bank_system_prompt": "",
        "knowledge_sources_filters": [],
        "knowledge_enable_auto_filtering": True,
        "knowledge_sources_displayed_metas": [],
        "knowledge_source_url": False,
        "knowledge_source_title": False,
        "knowledge_source_thumbnail": False,
        "permanent_delete": True,
        "upload_folder": UPLOAD_OLDER_ID,
        "enable_llm_citations": True,
        'log_level': 'INFO',
        "disclaimer": "The answers are generated by an artificial intelligence model. Please exercise human judgment and oversight to ensure all final outcomes are accurate, appropriate, and adhere to responsible use guidelines.",
        "http_headers": [
            {
                "from": "Content-Security-Policy",
                "to": "frame-ancestors \u0027self\u0027;"
            },
            {
                "from": "Strict-Transport-Security",
                "to": "max-age\u003d31536000; includeSubDomains"
            },
            {
                "from": "X-Frame-Options",
                "to": "SAMEORIGIN"
            },
            {
                "from": "X-Content-Type-Options",
                "to": "nosniff"
            }
        ],
        "max_thread_workers_number": 0,
        "filter_logged_sources": True,
        "n_top_sources_to_log": 20,
        "show_advanced_settings": True,
        "enable_smart_usage_of_kb": True,
        "knowledge_bank_description": "This knowledge bank contains informations about reviews done by our customers on the products we sell on our marketplace.",
        "use_custom_rebranding": False,
        "custom_theme_name": "",
        "custom_logo_file_name": "",
        "custom_icon_file_name": "",
        "user_profile_settings": [
            {"from": "user_id", "to": "user_id"},
            {"from": "country", "to": "country"},
            {"from": "department", "to": "department"},
        ],
        "include_user_profile_in_KB_prompt": True,
        "user_profile_dataset" : "user_profile_mysql",
        "max_n_upload_files": 5,
        "max_upload_size_mb": 10,
        "allow_doc_as_image": True,
        "docs_per_page_as_image": 5

    },
    "default_project_key": DEFAULT_PROJECT_KEY

}


def update(original, updates):
    # Update the default config with local config
    for key, value in updates.items():
        if isinstance(original.get(key), list) and isinstance(value, list):
            original[key] = value
        elif isinstance(original.get(key), dict) and isinstance(value, dict):
            original[key].update(value)
        else:
            original[key] = value
    return original


def load_config():
    local_config_path = Path(__file__).parent / "local_config.json"
    config = DEFAULT_CONFIG.copy()
    # Override with local configuration if it exists
    if local_config_path.exists():
        with open(local_config_path, 'r', encoding=locale.getpreferredencoding(False)) as local_config_file:
            local_config = json.load(local_config_file)
            # Update the default config with local config
            update(config, local_config)
            # config.update(local_config)
    else:
        print("No local configuration found. Default configuration will be used. Create a local_config.json file to override it.")
    return config


CONFIG = load_config()
os.environ["DKU_CURRENT_PROJECT_KEY"] = CONFIG.get("default_project_key", "")
os.environ["DKU_CUSTOM_WEBAPP_CONFIG"] = json.dumps(
    CONFIG.get("webapp_config"), ensure_ascii=False)


def get_setup_for_dataiku_client():
    return {
        "webapp_config": CONFIG.get("webapp_config"),
        "default_project_key": CONFIG.get("default_project_key"),
    }


dictConfig(
    {
        "version": 1,
        "formatters": {
            "default": {
                "format": "[%(asctime)s] %(levelname)s in %(module)s: %(message)s",
            }
        },
        "handlers": {
            "wsgi": {
                "class": "logging.StreamHandler",
                "stream": "ext://flask.logging.wsgi_errors_stream",
                "formatter": "default",
            }
        },
        "root": {"level": "INFO", "handlers": ["wsgi"]},
    }
)


def setup_dataiku_client():
    dataiku_setup = get_setup_for_dataiku_client()
    dataiku_api.setup(**dataiku_setup)