from typing import Dict, List, Literal, Tuple

from answers.backend.db.sql.dataset_schemas import (
    CONVERSATION_DATASET_CONF_ID,
    MESSAGE_DATASET_CONF_ID,
    ConfigDatasetID,
)
from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger
from flask import Blueprint, Response, jsonify, request
from marshmallow import Schema

checking = Blueprint("api_checking", __name__)

EndpointName = Literal["ask", "conversations"]

ENDPOINT_DEPENDENCIES: Dict[EndpointName, List[ConfigDatasetID]] = {
    "ask": [MESSAGE_DATASET_CONF_ID], # type: ignore
    "conversations": [MESSAGE_DATASET_CONF_ID, CONVERSATION_DATASET_CONF_ID] # type: ignore
    }


def get_api_dataset_status(dataset_id: ConfigDatasetID)->Tuple[bool, str]:
    dataset_is_set = dataiku_api.webapp_config.get(dataset_id, None) is not None
    status_msg = f"The dataset {dataset_id} has been set." if dataset_is_set else f"The dataset {dataset_id} has not been set."
    return dataset_is_set, status_msg


def get_endpoint_availability(endpoint_name: EndpointName)->Tuple[Response, int]:
    if not dataiku_api.webapp_config.get("enable_answers_api", False):
        return jsonify(error=f"The endpoint `{endpoint_name}` is not available: The option 'Enable usage of the Answers API' is disabled."), 500
    if endpoint_name in ENDPOINT_DEPENDENCIES.keys():
        for dataset_name in ENDPOINT_DEPENDENCIES[endpoint_name]:
            dataset_is_set, status_msg = get_api_dataset_status(dataset_name)
            if not dataset_is_set:
                return jsonify(error=f"The endpoint `{endpoint_name}` is not available: {status_msg}"), 500
        return jsonify(message="Endpoint available"), 200
    return jsonify(error=f"The endpoint '{endpoint_name}' is not supported!"), 500


def check_request_schema(request_data, expected_schema: Schema)->Tuple[Response, int]:
    if not isinstance(expected_schema, Schema):
        error_message = f"A wrong schema has been passed: The object `{expected_schema}` is of type `{type(expected_schema)}` while the variable "\
            f"`expected_schema` must be of type `{type(Schema)}`"
        logger.debug(error_message)
        raise Exception(error_message)
    if validation_errors := expected_schema.validate(request_data):
        check_log = f"Schema validation errors: {validation_errors}. Initial request: {request_data}."
        logger.debug(check_log)
        return jsonify(error=check_log), 400
    return jsonify(message="Request schema is valid"), 200


def check_get_params(*required_keys):
    missing = [key for key in required_keys if not request.args.get(key)]
    if missing:
        return f"Missing required GET parameters: {', '.join(missing)}"
    return None