from answers.solutions.chains.kb.generic_kb_agent import GenericKBAgent
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.llm_utils import handle_prompt_media_explanation


# DKUConversation
class DescriptionBasedKBAgent(GenericKBAgent):
    # Agent that decides if the kb is relevant to the last user input in the
    # chat history and the kb description and it generates a search query based on
    # those inputs and empty query if it is a terminal statement.
    def __init__(self, chat_has_media: bool = False):
        # Fetch the description of the knowledge bank from configuration.
        self.kb_description = dataiku_api.webapp_config.get("knowledge_bank_description")
        include_user_profile = dataiku_api.webapp_config.get("include_user_profile_in_KB_prompt", False)
        profile_examples = (
            [
                """
            # USER PROFILE:
            {
                "country": {
                    "value": "USA",
                    "description": "country"
                }
            }
            # --- END OF USER PROFILE ---
    """,
                """
            # USER PROFILE:
            {
                "department": {
                    "value": "IT",
                    "description": "the department of the user"
                }
            }
            # --- END OF USER PROFILE ---
    """,
                """
            # USER PROFILE:
            {
                "group": {
                    "value": "Data Scientists",
                    "description": "the LDAP group of the user"
                }
            }
            # --- END OF USER PROFILE ---
    """,
            ]
            if include_user_profile
            else ["", "", ""]
        )
        profile_example1_mention1 = "in the USA" if include_user_profile else ""
        profile_example1_mention2 = (
            "In addition the user profile indicates that the user is in the USA" if include_user_profile else ""
        )
        examples_template = """
        ### Example Scenarios:
            #### Example 1:
            {profile_example_1}
            [{{"role": "user", "content":"Hello"}},
            {{"role": "assistant", "content":"Hi there! How may I assist you today?"}},
            {{"role": "user", "content":"What is our policy on remote work?"}}]
            EXPECTED OUTPUT: {{
                "query": "Remote work policy {profile_example1_mention1} at my company.",
                "justification": "The user is asking about a policy covered by the Knowledge Bank. {profile_example1_mention2}"
            }}

            #### Example 2:
            {profile_example_2}            
            [{{"role": "user", "content":"Good morning"}},
            {{"role": "assistant", "content":"Good morning! How may I assist you today?"}},
            {{"role": "user", "content":"Just checking in, thanks!"}}]
            EXPECTED OUTPUT: {{
                "query": null,
                "justification": "The user's input is a terminal statement and does not require further information."
            }}

            #### Example 3:
            {profile_example_3}
            [{{"role": "user", "content":"Hello"}},
            {{"role": "assistant", "content":"Hello! How may I assist you today?"}},
            {{"role": "user", "content":"count to 20"}},
            {{"role": "assistant", "content":"1, 2, 3, ..., 20"}},
            {{"role": "user", "content":"Thanks!"}}]
            EXPECTED OUTPUT: {{
                "query": null,
                "justification": "The user's input is a terminal statement and does not require further information."
            }}
        """
        examples = examples_template.format(
            profile_example_1=profile_examples[0],
            profile_example_2=profile_examples[1],
            profile_example_3=profile_examples[2],
            profile_example1_mention1=profile_example1_mention1,
            profile_example1_mention2=profile_example1_mention2,
        )
        user_profile_mention = ", user profile" if include_user_profile else ""
        system_prompt = f"""
        You are a search query builder for a knowledge bank. Your role is to analyze the chat history{user_profile_mention} and last user input and decide whether additional information retrieval is necessary, based solely on the content available in the Knowledge Bank.
        In the case where user provided documents or images during the conversation, consider them as well. In that case, the documents/images would have been uploaded to a webapp and you would have the extracted texts and or images.
        If for any reason, you don't have any information about the provided documents/images, ignore them and don't mention it to user.
        # Knowledge Bank Overview
        The knowledge bank is a vector database designed to aid in answering questions by providing relevant information. Below is the detailed description:
        ## Knowledge Bank's Description:
        {self.kb_description}
        ## -- End of Description --

        # Your Task
        Analyze the chat history and Knowledge Bank description. If the chat suggests the need for more information on topics covered by the Knowledge Bank, produce a search query based on the decision criteria outlined below. Produce the response as a JSON object with two fields: 'query' and 'justification'. If no query is needed, 'query' should be null and 'justification' should explain why no query is necessary. Provide the JSON object only. Do not add any other explanations, decorations, or additional information.

        # Decision Criteria:
        - Relevant Inquiry: Generate a search query if the latest input from the user in the chat history implies a need for more information on a specific topic that aligns with the description of the Knowledge Bank.

        - Response Protocol: If a last user input does not directly require information from the Knowledge Bank, the 'query' should be null. This is critical to ensure that the system remains focused and only retrieves or provides information when it is relevant to do so.

        - Irrelevant or Terminal Input: If the last input from the user in the chat history is a brief greeting, a closure statement, or a question that falls outside the scope of the Knowledge Bank's description, respond with 'query' as null and provide a justification.
        {examples}
        """
        user_profile_section = (
            """
        # USER PROFILE:
        The following is the user’s profile, which provides context about the user:
        {user_profile}
        # --- END OF USER PROFILE ---        
        """
            if include_user_profile
            else ""
        )

        self._system_prompt = handle_prompt_media_explanation(system_prompt=system_prompt, has_media=chat_has_media)
        user_profile_mention2 = (
            "Consider information provided by the user profile and, if meaningful, take int into account while crafting the search query."
            if include_user_profile
            else ""
        )
        self._user_prompt = """
        Evaluate the following chat history to determine if a semantic search query should be generated. Generate the response as a JSON object with two fields: 'query' and 'justification'. 
        Generate a search query only if there is a need for more information on topics explicitly covered by the Knowledge Bank. 
        {user_profile_mention}
        If no query is needed, 'query' should be null and 'justification' should explain why no query is necessary. 
        Provide the JSON object only. Do not add any other explanations, decorations, or additional information beyond the JSON object.
        {user_profile_section}
        """.format(user_profile_section=user_profile_section, user_profile_mention=user_profile_mention2)


    @property
    def system_prompt(self):
        return self._system_prompt

    @property
    def user_prompt(self):
        return self._user_prompt