import time
from typing import Any, Dict, Optional, Union

from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger
from typing_extensions import Literal

ONE_WEEK_IN_SECONDS = 604800
ImageSettingType = Literal["height", "width", "quality", "nbr_images_to_generate"]


def time_has_passed(reference_time: int, time_threshold: int = ONE_WEEK_IN_SECONDS) -> bool:
    return int(time.time()) - reference_time > time_threshold


def get_num_images_user_can_generate(user_profile: Optional[Dict[str, Any]] = None) -> int:
    config: Dict[str, str] = dataiku_api.webapp_config
    max_images_to_generate = int(config.get("max_images_per_user_per_week", 0))
    if not user_profile:
        return max_images_to_generate
    generated_images_info = get_generated_images_info(user_profile)
    current_count = int(generated_images_info.get("count", 0))
    current_time = int(time.time())
    first_generated_at = generated_images_info.get("first_generated_at", current_time)
    logger.debug(f"User generated {current_count} images out of {max_images_to_generate} images")
    logger.debug(f"user generated first image at {first_generated_at}")
    nbr_images_to_generate = (
        user_profile.get("media", {}).get("image", {}).get("nbr_images_to_generate", {}).get("value", 0)
    )
    if max_images_to_generate and (current_count >= max_images_to_generate and not time_has_passed(first_generated_at)):
        logger.info("User has reached the maximum number of images that can be generated")
        return 0
    if time_has_passed(first_generated_at):
        logger.debug("User can generate the maximum number of images as the week has passed")
        return max_images_to_generate
    if current_count + nbr_images_to_generate > max_images_to_generate:
        logger.debug(f"User can generate {max_images_to_generate - current_count} images less than max")
        return max_images_to_generate - current_count
    logger.debug(f"User can generate {nbr_images_to_generate} images")
    return max_images_to_generate


def update_user_profile_generated_media_info(user_profile: Dict[str, Any], nbr_images_generated: int) -> Dict[str, Any]:
    current_time = int(time.time())

    # Fetch current generated images data
    generated_images_info = get_generated_images_info(user_profile)
    current_count = generated_images_info.get("count", 0)
    first_generated_at_time = generated_images_info.get("first_generated_at", current_time)

    # Calculate the new count
    new_count = current_count + nbr_images_generated

    # Check if the first generated image is older than a week (604800 seconds)
    if current_count == 0:  # First image generation
        generated_images_info = {"count": nbr_images_generated, "first_generated_at": current_time}
    elif time_has_passed(first_generated_at_time):
        # Reset count if a week has passed since the first generation
        generated_images_info = {"count": nbr_images_generated, "first_generated_at": current_time}
        new_count = nbr_images_generated
    else:
        # Update only the count if a week hasn't passed
        generated_images_info["count"] = new_count
    user_profile["generated_media_info"] = {"image": generated_images_info}
    return user_profile


def get_generated_images_info(user_profile: Dict[str, Any]) -> Union[Dict[str, Any], Any]:
    media = user_profile.get("generated_media_info", {})
    return media.get("image", {})

def get_image_setting_value(user_profile: Dict[str, Any], setting: ImageSettingType) -> Optional[Any]:
    image_settings = user_profile.get("media", {}).get("image")
    if setting not in image_settings or not image_settings[setting].get("value"):
        return None
    return image_settings.get(setting)["value"]

def get_image_height(user_profile: Dict[str, Any]) -> Optional[int]:
    height = get_image_setting_value(user_profile, "height")
    return int(height) if height else None 


def get_image_width(user_profile: Dict[str, Any]) -> Optional[int]:
    width = get_image_setting_value(user_profile, "width")
    return int(width) if width else None


def get_image_quality(user_profile: Dict[str, Any]) -> Optional[str]:
    quality = get_image_setting_value(user_profile, "quality")
    return str(quality) if quality else None


def get_nbr_images_to_generate(user_profile: Dict[str, Any]) -> int:
    nbr_images_to_generate = get_image_setting_value(user_profile, "nbr_images_to_generate")
    return int(nbr_images_to_generate) if nbr_images_to_generate else 1


def set_nbr_images_to_generate(user_profile: Dict[str, Any], nbr_images: int) -> Dict[str, Any]:
    if (
        "media" not in user_profile
        or "image" not in user_profile["media"]
        or "nbr_images_to_generate" not in user_profile["media"]["image"]
    ):
        return user_profile
    logger.debug(f"Setting number of images to generate to {nbr_images}")
    user_profile["media"]["image"]["nbr_images_to_generate"]["value"] = nbr_images
    return user_profile


def flatten_user_profile_values(user_profile: Optional[Dict[str, Any]]):
    """Processes the user profile coming from a frontend request to flatten all the custom setting values
    Example user profile coming from the frontend ('Role' and 'Country' are the custom settings): 
    user_profile = {
        "language": "English", 
        "Role": {"value": "Developer", "description": "Your role"}, 
        "Country": {"value": "France", "description": "Your living country"}
        }
    """
    reformated_user_profile = {}
    if isinstance(user_profile, dict):
        for setting in user_profile.keys():
            setting_value = user_profile.get(setting, "")
            if (isinstance(setting_value, dict)) and ("value" in setting_value) and ("description" in setting_value) and (len(setting_value.keys())==2):
                setting_value = setting_value["value"]
            reformated_user_profile[setting] = setting_value
        return reformated_user_profile
    return user_profile