import { ServerApi } from '@/api/server_api'
import i18n from '@/i18n'
import { ref } from 'vue'
import { ResponseStatus, RetrievalMode, type FilterConfig, type RetrieverInfo, UploadableFileTypes } from '@/models'
import { WT1iser } from '@/common/wt1'
import { createNotification, stringToEnum } from '@/common/utils'

interface UISetup {
  examples: string[]
  title?: string
  subtitle?: string
  questionPlaceholder?: string
  language?: string
  projectKey?: string
  docsFolderId?: string
  uploadFolderId?: string
  uploadFileTypes?: string
  selfServiceEmbeddingModel?: string | null
  feedbackPositiveChoices?: string[]
  feedbackNegativeChoices?: string[]
  retrievalMode?: RetrievalMode
  retriever?: RetrieverInfo | null
  displaySqlQuery?: boolean
  sqlRetrievalConnection?: string
  sqlRetrievalSuggestedJoins?: string[]
  llmCapabilities?: { streaming: boolean; multiModal: boolean }
  llmId?: string
  imageGenerationLLMId?: string
  filtersConfig?: FilterConfig | null
  user?: string
  currentUserProfile?: Record<string, any> | null
  userSettings?: Record<string, string> | null
  supportedLanguages?: any[] | null
  allowGeneralFeedback?: boolean
  disclaimer?: string
  customThemeName?: string
  useCustomRebranding?: boolean
  customLogoFileName?: string
  customIconFileName?: string
  displayChunks?: boolean,
  maxUploadSizeMB?: number
  maxNUploadFiles?: number
  maxImagesToGenerate?: number
  allowMediaConversation?: boolean
}

const infoTextOne = '“Lorem ipsum dolor sit amet, consectetur adipiscing elit”'
const infoTextTwo =
  '“Fusce pulvinar augue quis commodo pharetra, quisque mattis, diam ut tempor egestas”'
const infoTextThree =
  '“Maecenas augue nunc, fermentum ac placerat sed, eleifend eget odio. Nunc ut cursus felis”'

const initExamples = [infoTextOne, infoTextTwo, infoTextThree]

const initUISetup: UISetup = {
  examples: initExamples
}

const setup = ref<UISetup>(initUISetup)

export function useUI() {
  async function initializeUI() {
    try {
      const response = await ServerApi.getUISetup()
      if (response && response.data) {
        const retriever = response.data.retriever_info
          ? {
              id: response.data.retriever_info.id,
              name: response.data.retriever_info.name,
              alias: response.data.retriever_info.alias,
              activated: response.data.retriever_info.activated,
              type: response.data.retriever_info.type
            }
          : null
        const currentUserProfile = retrieveCurrentUserProfile(response);
        const llmCapabilities = {
          streaming: response.data.llm_capabilities.streaming,
          multiModal: response.data.llm_capabilities.multi_modal
        }
        let uploadFileTypes = ''
        // TODO: this will change to once in memory RAG is supported.
        // We'll need to look for an embedding model id.
        if (response.data?.upload_folder_id && response.data.max_n_upload_files > 0) {
          uploadFileTypes += UploadableFileTypes.DOCUMENT
          if (llmCapabilities.multiModal && response.data.upload_folder_id) {
            uploadFileTypes += UploadableFileTypes.IMAGE
          }
        }

        const uiSetup: UISetup = {
          examples: response.data.examples || [],
          title: response.data.title,
          subtitle: response.data.subtitle,
          questionPlaceholder: response.data.input_placeholder,
          projectKey: response.data.project,
          docsFolderId: response.data.docs_folder_id,
          uploadFolderId: response.data.upload_folder_id,
          uploadFileTypes: uploadFileTypes,
          selfServiceEmbeddingModel: response.data.self_service_embedding_model,
          feedbackPositiveChoices: response.data.feedback_positive_choices,
          feedbackNegativeChoices: response.data.feedback_negative_choices,
          filtersConfig: response.data.filters_config,
          retrievalMode: stringToEnum(RetrievalMode, response.data.retrieval_mode),
          retriever: retriever,
          displaySqlQuery: response.data.display_sql_query,
          sqlRetrievalConnection: response.data.sql_retrieval_connection,
          sqlRetrievalSuggestedJoins: response.data.sql_retrieval_suggested_joins,
          displayChunks: response.data.display_source_chunks ?? true,
          useCustomRebranding:
            response.data.use_custom_rebranding && response.data.custom_theme_name !== '',
          customThemeName: response.data.custom_theme_name,
          customLogoFileName: response.data.custom_logo_file_name,
          customIconFileName: response.data.custom_icon_file_name,
          llmCapabilities: llmCapabilities,
          llmId: response.data.llm_id,
          imageGenerationLLMId: response.data.image_generation_llm_id,
          user: response.data.user,
          currentUserProfile: currentUserProfile,
          userSettings: response.data.current_user_profile,
          supportedLanguages: response.data.supported_languages,
          language: response.data.language,
          allowGeneralFeedback: response.data.allow_general_feedback,
          disclaimer: response.data.disclaimer,
          maxUploadSizeMB: response.data.max_upload_size_mb,
          maxNUploadFiles: response.data.max_n_upload_files,
          maxImagesToGenerate: response.data.max_images_per_user_per_week
        }

        if (Object.keys(i18n.global.messages.value).includes(response.data.language)) {
          /* eslint-disable @typescript-eslint/ban-ts-comment */
          // @ts-ignore
          i18n.global.locale.value = response.data.language
        }

        setup.value = uiSetup
      } else {
        throw new Error('Error initializing UI')
      }
    } catch (error) {
      WT1iser.error('initializeUI', 0, 'Error initializing UI') // TODO: Add error message
      createNotification('negative', 'Error starting the application')
    }
  }
  function retrieveCurrentUserProfile(response: any) {
    const defaultUserLang = response.data.default_user_language
    const currentLang =  defaultUserLang ?? response.data.supported_languages?.find((l: any) => l.key === response.data.language)?.value
        const currentUserProfile: Record<string, any> = {
          language: {
            value: currentLang ?? 'English',
            description: "User's language"
          }
        }
        const mediaGenerationOn = response.data.llm_capabilities.image_generation
        if (mediaGenerationOn) {
          currentUserProfile.media = {
            image: {
              height: {
                value: '',
                description: 'height_description',
                options: [256, 512, 1024, 1792, 2048],
                clearable: true
              },
              width: {
                value: '',
                description: 'width_description',
                options: [256, 512, 1024, 1792, 2048],
                clearable: true
              },
              quality: {
                value: '',
                description: 'quality_description',
                options: ['standard', 'hd'],
                clearable: true
              },
              nbr_images_to_generate: {
                value: 1,
                description: 'nbr_images_to_generate_description',
                options: [1, 2, 3, 4],
                clearable: false
              }
            }
          }
        }
        if (response.data.user_settings) {
          for (const s of response.data.user_settings) {
            currentUserProfile[s.from] = { value: '', description: s.to }
            currentUserProfile[s.from].value = response.data.current_user_profile
              ? response.data.current_user_profile[s.from] ?? ''
              : ''
          }
          if (
            response.data.current_user_profile &&
            response.data.current_user_profile['language']
          ) {
            currentUserProfile['language'].value = response.data.current_user_profile['language']
          }
          if (
            mediaGenerationOn &&
            response.data.current_user_profile &&
            response.data.current_user_profile['media']
          ) {
            for (const key in currentUserProfile.media.image) {
              currentUserProfile.media.image[key].value =
                response.data.current_user_profile['media']['image'][key].value ?? ''
            }
            if(response.data.current_user_profile['generated_media_info']){
              currentUserProfile['generated_media_info'] = response.data.current_user_profile['generated_media_info']
            }
        }
      }
        return currentUserProfile
  }
  async function updateUserProfile(settings: any, successMessage: string, failureMessage: string) {
   const settingsValues = Object.fromEntries(
      Object.entries(settings).map(([k, v]) => [k, (v as any).value == null ? (v as any): (v as any).value])
    )
    for (const key in setup.value.userSettings) {
      // Empty core user profile settings are updated from the initial 'setup':
      if (settingsValues[key]==null) {
        settingsValues[key] = setup.value.userSettings[key]
      }
    }

    const response = await ServerApi.updateUserProfile({ profile: settingsValues })
    if (response && response.status === ResponseStatus.OK) {
      setup.value.currentUserProfile = settings
      createNotification('positive', successMessage)
    } else {
      createNotification('negative', failureMessage)
    }
  }
  function isSmallScreen() {
    return (
      window.innerWidth <= 600 ||
      (screen.orientation.type.indexOf('landscape') >= 0 &&
        window.innerWidth <= 1000 &&
        window.innerHeight <= 600)
    )
  }
  return {
    initializeUI,
    setup,
    isSmallScreen,
    updateUserProfile
  }
}