llmApp.service('customFileUploaderService', ['$q', '$timeout', function ($q, $timeout) {
    const defaultIconURL = `${window.location.origin}/plugins/document-question-answering/resource/custom-ui-settings/No_image_available.png`;

    // Add UUID generation function
    this.generateAppFilename = function () {
        const url = window.location.href;
        const match = url.match(/projects\/([^\/]+)\/webapps\/([^\/]+)\/edit/);

        if (match) {
            const projectName = match[1]; // "TUT_LLM_MULTIMODALEMBEDDING"
            const webappPart = match[2]; // "b5v2w4z_test-webapp-custom"
            const parts = webappPart.split('_');

            if (parts.length >= 2) {
                const id = parts[0]; // "b5v2w4z"
                const name = parts.slice(1).join('_'); // "test-webapp-custom"
                return `${projectName}_${name}_${id}`;
            }
        }

        // Fallback if URL doesn't match expected pattern
        return 'unknown-project_unknown-app_unknown-id';
    };

    // File type utilities
    this.isImageFile = function (file) {
        return file && file.type && file.type.startsWith('image/');
    };

    this.isImageFileByName = function (fileName) {
        const imageExtensions = ['.jpg', '.jpeg', '.png', '.gif', '.svg', '.webp', '.bmp'];
        const extension = (fileName || '').toLowerCase().substring(fileName.lastIndexOf('.'));
        return imageExtensions.includes(extension);
    };

    this.getDefaultIconUrl = function () {
        return defaultIconURL;
    };

    // Folder structure operations
    this.ensureSubFolderExists = function (uploadTargetType, uploadTargetPath, subFolderName) {
        const subFolderPath = `${uploadTargetPath.replace(/\/$/, '')}/${subFolderName}`;
        return DataikuAPI.admin.folderEdit.createContent(uploadTargetType, subFolderPath, true)
            .then(() => subFolderPath)
            .catch(error => {
                if (error.data?.detailedMessage?.includes('already exists')) {
                    return subFolderPath;
                }
                throw error;
            });
    };



    this.ensureThemeFolderExists = function (uploadTargetType, themeName, uploadTargetPath) {
        const themePath = `${uploadTargetPath}/${themeName}`;
        return DataikuAPI.admin.folderEdit.createContent(uploadTargetType, themePath, true)
            .then(() => themePath)
            .catch(error => {
                if (error.data?.detailedMessage?.includes('already exists')) {
                    return themePath;
                }
                throw error;
            });
    };

    this.ensureSubFolderExists = function (uploadTargetType, themePath, folderName) {
        const subFolderPath = `${themePath}/${folderName}`;
        return DataikuAPI.admin.folderEdit.createContent(uploadTargetType, subFolderPath, true)
            .then(() => subFolderPath)
            .catch(error => {
                if (error.data?.detailedMessage?.includes('already exists')) {
                    return subFolderPath;
                }
                throw error;
            });
    };

    // File operations
    this.fetchUploadedFiles = function (uploadTargetType) {
        return DataikuAPI.admin.folderEdit.listContents(uploadTargetType);
    };

    this.listSubFolders = function (folderName = 'answers') {
        return this.fetchUploadedFiles('LOCAL_STATIC').then(
            response => {
                var data = typeof response === 'string' ? JSON.parse(response) : response;
                // Find the specified folder in the data array
                var targetFolder = data.data.find(function(item) {
                    return item && item.name === folderName;
                });
                // Check if target folder exists and has children
                if (!targetFolder || !targetFolder.children) {

                    return [];
                }
                // Process each child of the target folder that has children property
                var result = targetFolder.children
                    .filter(function(child) {
                        return child && child.children;
                    })
                    .map(function(child) {
                        return {
                            label: child.name,
                            value: child.name
                        };
                    });
                return result;
            }
        ).catch(error => {

                return [];
            }
        );
    }


    this.uploadFile = function (uploadTargetType, targetPath, file) {
        return new $q((resolve, reject) => {
            DataikuAPI.admin.folderEdit.uploadContent(uploadTargetType, targetPath, file, (response, error) => {
                if (error) {
                    reject({file: file.name, error});
                } else {
                    resolve({
                        file: file.name,
                        path: `${targetPath}/${file.name}`
                    });
                }
            });
        });
    };

    this.deleteFile = function (uploadTargetType, filePath) {
        return DataikuAPI.admin.folderEdit.deleteContent(uploadTargetType, filePath);
    };


    this.getAcceptedImageTypes = function () {
        return ['image/jpeg', 'image/jpg', 'image/png'];
    };

    this.isAcceptedImageType = function (fileType) {
        const acceptedTypes = this.getAcceptedImageTypes();
        return acceptedTypes.includes(fileType);
    };


    // File validation
    this.getDefaultAllowedFiles = function () {
        return {
            'custom.css': {types: ['text/css', 'application/css'], folder: ''},
            'fonts.css': {types: ['text/css', 'application/css'], folder: 'fonts'},
            'image': {types: ['image/png', 'image/jpeg'], folder: 'images'}
        };
    };

// Add this method to customFileUploaderService
    this.createFolder = function(uploadTargetType, folderName, parentPath) {
        // First ensure the answers folder exists
        return this.ensureSubFolderExists(uploadTargetType, parentPath,'answers')
            .then(answersPath => {
                // Now create the new folder inside the answers folder
                const folderPath = `${answersPath}/${folderName}`;
                return DataikuAPI.admin.folderEdit.createContent(uploadTargetType, folderPath, true)
                    .then(() => {
                        // Return the folder in the format expected by folder-selector
                        return {
                            label: folderName,
                            value: folderName
                        };
                    });
            });
    };

    this.getAllFilesForTheme = function(apiResponseData, themeName, parentFolderName = 'answers') {
        if (!apiResponseData || !themeName) {
            return [];
        }

        // 1. Find the top-level parent folder.
        const parentFolder = apiResponseData.find(item => item.name === parentFolderName);
        if (!parentFolder || !parentFolder.children) {
            // Parent folder doesn't exist or is empty.
            return [];
        }

        // 2. Find the specific theme folder within the parent folder.
        const themeFolder = parentFolder.children.find(child => child.name === themeName);
        if (!themeFolder || !themeFolder.children) {
            // The specific theme folder doesn't exist or is empty.
            return [];
        }

        // 3. Use a recursive helper function to collect all files within the theme folder.
        const files = [];
        const collectFilesRecursively = (items) => {
            items.forEach(item => {
                // If an item has children, it's a sub-folder; recurse into it.
                if (item.children && item.children.length > 0) {
                    collectFilesRecursively(item.children);
                }
                    // If an item has no 'children' array, or an empty one, we treat it as a file.
                    // The API sometimes represents empty folders with `children: []`.
                // Files are typically represented by not having a `children` property at all.
                else if (!item.children) {
                    files.push(item);
                }
            });
        };

        collectFilesRecursively(themeFolder.children);
        return files;
    };



}]);
