from typing import Any, Dict, List, Optional

from answers.backend.db.logging import logging_sql_manager
from answers.backend.db.user_profile import user_profile_sql_manager
from answers.solutions.knowledge_bank import get_knowledge_bank_full_name
from answers.solutions.service import MEDIA_CONVERSATION_START_TAG, LLM_Question_Answering
from common.backend.models.base import ConversationType, ExtractedQueryInfo, GeneratedMedia, MediaSummary, QuestionData
from common.backend.services.sources.sources_formatter import format_sources_to_store
from common.backend.utils.context_utils import get_main_trace_dict
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.image_generation_utils import handle_images
from common.llm_assist.logging import logger
from common.solutions.chains.title.media_conversation_title import SummaryTitler


def record_conversation(user: str, info: ExtractedQueryInfo) -> Optional[Dict[str, Any]]:
    """
    Extracts data from a JSON request for DB updating purposes. It also includes additional details
    like conversation name and knowledge bank identifiers when necessary.

    Parameters:
    - info (ExtractedQueryInfo): The JSON data of the request with potential keys for query, filters, answer, etc.

    Notes:
    - Successfully updates the logging database with new interaction information and confirms with a success message.
    - Used to track user interactions and LLM responses during specific events such as new queries or at conversation closure.
    """
    try:
        query = info["query"] if info["query"] != MEDIA_CONVERSATION_START_TAG else ""
        answer = info["answer"]
        conversation_name: Optional[str] = info["conversation_name"]
        conversation_id: Optional[str] = info["conversation_id"]
        conversation_type: Optional[str] = info["conversation_type"]
        if not (llm_context := info.get("llm_context", {})):
            llm_context = {}
        if not conversation_id:
            if conversation_type == ConversationType.GENERAL:
                conversation_name = LLM_Question_Answering.get_conversation_title(info["query"], answer, info.get("user_profile", None))
            elif conversation_type == ConversationType.MEDIA_QA:
                media_qa_context: List[MediaSummary] = llm_context.get("media_qa_context", [])
                conversation_name = SummaryTitler().generate_summary_title(media_qa_context)
            else:
                raise Exception("Unknown Conversation type.")
        llm_context["trace"] = get_main_trace_dict(query, answer)
        knowledge_bank_full_id = get_knowledge_bank_full_name(info["knowledge_bank_id"])

        # Get llm id
        config: Dict[str, str] = dataiku_api.webapp_config
        llm_id = config.get("llm_id", None)  # Need to check
        if fallback_llm := config.get("fallback_llm"):
            llm_context["fallback_llm"] = fallback_llm
        images: List[GeneratedMedia] = []
        user_profile = info.get("user_profile", {})
        
        generated_images = info.get("generated_images", [])
        if generated_images:
            images, user_profile = handle_images(generated_images, user, user_profile, user_profile_sql_manager)
        
        # Sources handling
        aggregated_sources_to_store = format_sources_to_store(info["sources"])

        # Update cache conversation
        new_history_record = QuestionData(
            id=str(len(info["history"])),
            query=query,
            filters=info["filters"],
            file_path=None,  # TODO: legacy: to remove
            answer=answer,
            sources=aggregated_sources_to_store,
            feedback=None,
            llm_context=llm_context,
            generated_media={"images": images},
        )

        record_id, conversation_infos = logging_sql_manager.add_record(
            dict(new_history_record),
            user,
            conversation_id,
            conversation_name,
            knowledge_bank_full_id,  # TODO: remove once we change to new logging structure
            llm_id,
        )

        return {
            "record_id": record_id,
            "conversation_infos": conversation_infos,
            "images": images,
            "user_profile": user_profile,
            "new_record": new_history_record,
        }

    except Exception as e:
        logger.exception(f"Error storing conversation in DB {e}")
        return None
