from functools import lru_cache
from typing import Any, Dict, List, Optional

from answers.backend.models.base import SUPPORTED_METADATA_DATATYPES, KnowledgeBankInfo
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.dataiku_utils import find_antecedent_recipe
from common.llm_assist.logging import logger
from dataiku.core.knowledge_bank import KnowledgeBank
from dataiku.core.project import Project
from dataikuapi.dss.knowledgebank import DSSKnowledgeBankListItem

webapp_config = dataiku_api.webapp_config


@lru_cache(maxsize=None)
def get_knowledge_bank_full_name(knowledge_bank_id: str):
    if knowledge_bank_id:
        knowledge_bank = KnowledgeBank(knowledge_bank_id, project_key=dataiku_api.default_project_key)
        return knowledge_bank.full_name
    else:
        return None


@lru_cache(maxsize=None)
def get_core_knowledge_bank(knowledge_bank_id: str):
    if knowledge_bank_id:
        return dataiku_api.default_project.get_knowledge_bank(knowledge_bank_id).as_core_knowledge_bank()
    else:
        return None


@lru_cache(maxsize=None)
def get_knowledge_bank_name(id: Optional[str]) -> Optional[str]:
    if id is None:
        return None
    project = dataiku_api.default_project

    short_id = id
    if "." in id:
        (project_key, short_id) = id.split(".", 1)
    for kb in project.list_knowledge_banks():
        item: DSSKnowledgeBankListItem = kb
        if item.id == short_id:
            name: str = kb.name
            return name
    return None


def get_knowledge_bank_retriever(
    knowledge_bank_id: Optional[str] = None,
    filters: Optional[Dict[str, List[Any]]] = None,
    search_type: Optional[str] = "",
) -> Any:
    from answers.backend.utils.parameter_helpers import get_retriever_search_kwargs, load_knowledge_bank_parameters
    knowledge_bank_parameters, __ = load_knowledge_bank_parameters()
    retrieval_parameters = knowledge_bank_parameters["retrieval_parameters"]
    project = dataiku_api.default_project
    vector_db_type = get_vector_db_type(project, knowledge_bank_id)
    retriever_search_kwargs = get_retriever_search_kwargs(retrieval_parameters, vector_db_type, filters)

    logger.info(f"""
    Vector DB type: {vector_db_type}
    Search type: {search_type}
    Retriever search kwargs: {retriever_search_kwargs}
    """)
    kb = KnowledgeBank(knowledge_bank_id, project_key=dataiku_api.default_project_key)
    if vector_db_type == "AZURE_AI_SEARCH":
        logger.debug(f"""Using Azure AI Search k value of {retrieval_parameters["k"]}""")
        retriever = kb.as_langchain_retriever(
        search_type=search_type,
        search_kwargs=retriever_search_kwargs,
        k=retrieval_parameters["k"]
    )
    else:
        retriever = kb.as_langchain_retriever(
        search_type=search_type,
        search_kwargs=retriever_search_kwargs,
    )
    return retriever


@lru_cache(maxsize=None)
def get_vector_db_type(project: Project, knowledge_bank_id: Optional[str] = None):
    if knowledge_bank_id:
        return project.get_knowledge_bank(knowledge_bank_id).as_core_knowledge_bank()._get()["vectorStoreType"]
    return None


@lru_cache(maxsize=None)
def get_knowledge_bank_info(project: Project, knowledge_bank_id: str)->KnowledgeBankInfo:
    graph = project.get_flow().get_graph()
    flow_nodes = graph.nodes
    kb_recipe_node = None

    try:
        kb_recipe_node = find_antecedent_recipe(graph.data, knowledge_bank_id)
    except KeyError:
        raise KeyError(f"The knowledge bank id `{knowledge_bank_id}` antecedent recipe could not be found")
    
    # Recipe params and predecessors:
    embedding_recipe_name = kb_recipe_node.get("ref")
    embedding_recipe_type = kb_recipe_node.get("subType")    
    recipe_predecessors = kb_recipe_node.get("predecessors")
    metadata_dataset_id = ""
    documents_folder_id = ""

    knowledge_bank_settings = project.get_knowledge_bank(knowledge_bank_id).get_settings()
    for predecessor_id in recipe_predecessors:
        if flow_nodes[predecessor_id]["type"] == "COMPUTABLE_DATASET":
            metadata_dataset_id = predecessor_id
        elif flow_nodes[predecessor_id]["type"] == "COMPUTABLE_FOLDER":
            documents_folder_id = predecessor_id
    metadata_schema = []
    for metadata_info in knowledge_bank_settings.get_raw().get("metadataColumnsSchema", []):
        metadata_datatype = metadata_info.get("type")
        if metadata_datatype in SUPPORTED_METADATA_DATATYPES:
            metadata_schema.append(metadata_info)
    kb_origin_info = KnowledgeBankInfo(embedding_recipe_name=embedding_recipe_name,
                                       embedding_recipe_type=embedding_recipe_type,
                                       vector_store_type=knowledge_bank_settings.get_raw()["vectorStoreType"],
                                       metadata_dataset_id=metadata_dataset_id,
                                       documents_folder_id=documents_folder_id,
                                       metadata_schema=metadata_schema)
    return kb_origin_info