from enum import Enum
from typing import Any, Dict, List, Optional, TypedDict, Union

from common.backend.models.source import AggregatedToolSources, Source
from langchain.chains import ConversationalRetrievalChain, ConversationChain

# TODO: probably feedback needs to be handeled differently, not stored in cache



class FeedbackValue(str, Enum):
    NEGATIVE = "NEGATIVE"
    POSITIVE = "POSITIVE"


class Feedback(TypedDict):
    value: FeedbackValue
    choice: List[str]
    message: str


class GeneratedMedia(TypedDict, total=False):
    file_data: str
    file_path: str
    file_format: str
    referred_file_path: str


class MediaSummary(TypedDict, total=False):
    original_file_name: str
    file_path: str
    metadata_path: str
    summary: Optional[str]
    topics: Optional[List[str]]
    questions: Optional[List[str]]
    full_extracted_text: Optional[str]
    preview: Optional[str]
    chain_type: Optional[str]
    token_estimate: Optional[int]
    is_deleted: Optional[bool]
    extracted_images_path: Optional[List[str]]
    begin_time: Optional[int]
    trace: Optional[Dict[str, Any]]


class ConversationType(str, Enum):
    GENERAL = "general"
    MEDIA_QA = "media_qa"


class ConversationInfo(TypedDict, total=False):
    id: str
    name: str
    timestamp: str  # todo remove timestamp and keep created_at + updated_at
    created_at: str
    updated_at: str


class LlmHistory(TypedDict):
    input: str
    output: str


class ConversationParams(TypedDict, total=False):
    user_query: str
    chain_type: Optional[str]
    media_summaries: Optional[List[MediaSummary]]
    previous_media_summaries: Optional[List[MediaSummary]]
    retrieval_enabled: bool
    knowledge_bank_selection: List[str]
    chat_history: List[LlmHistory]
    kb_query: Optional[str]
    db_query: Optional[Dict]
    use_db_retrieval: bool
    filters: Optional[Dict[str, List[Any]]]
    qa_chain: Union[ConversationChain, ConversationalRetrievalChain]
    global_start_time: float
    llm_capabilities: Dict[str, bool]
    justification: str
    user_profile: Optional[Dict[str, Any]]
    conversation_type: ConversationType
    self_service_decision: Optional[Dict[str, Union[str, List[str]]]]
    chain_purpose: Optional[str]
    user_agent: Optional[str]
    app_id: Optional[str]
    conversation_id: Optional[str]


class LLMStep(Enum):
    COMPUTING_PROMPT_WITH_KB = 1
    COMPUTING_PROMPT_WITH_DB = 2
    COMPUTING_PROMPT_WITHOUT_RETRIEVAL = 3
    STREAMING_START = 4
    STREAMING_END = 5
    STREAMING_ERROR = 6
    QUERYING_LLM_WITHOUT_RETRIEVAL = 7
    QUERYING_LLM_WITH_KB = 8
    QUERYING_LLM_WITH_DB = 9
    GENERATING_IMAGE = 10
    CALLING_AGENT = 11
    RETRIEVING_AGENT = 12
    NO_AGENT = 13
    AGENT_AVAILABLE = 14
    SYNTHESIZING_AGENTS_ANSWERS = 15
    CALLING_DEFAULT_CHAIN = 16
    PREPARING_AGENT = 17
    USING_FALLBACK_LLM = 18


class RetrieverMode(Enum):
    KB = "kb"
    DB = "db"
    NO_RETRIEVER = "no_retrieval"


class RetrieverInfo(TypedDict):
    name: Optional[str]
    alias: Optional[str]
    activated: bool
    type: str
    id: Optional[str]


class LLMStepDesc(TypedDict):
    step: LLMStep


class DatasetContext(TypedDict):
    sql_retrieval_table_list: Union[str, None]
    tables_used: List
    sql_query: str


class LLMContext(TypedDict, total=False):
    selected_retrieval_info: Union[RetrieverInfo, None]
    dataset_context: DatasetContext
    user_profile: Dict[str, Any]
    uploaded_docs: List
    media_qa_context: List
    llm_kb_selection: List[str]
    trace: Dict[str, Any]
    fallback_llm: Optional[str]


class RetrievalSummaryJson(TypedDict, total=False):
    answer: str
    sources: Union[List[AggregatedToolSources], List[Any]]
    generated_images: List
    filters: Optional[Dict[str, List[Any]]]
    knowledge_bank_selection: List
    llm_context: LLMContext
    user_profile: Dict[str, Any]


class QuestionData(TypedDict, total=False):
    id: str
    query: str
    filters: Optional[Dict[str, List[Union[str, float, int]]]]
    file_path: Optional[str]
    chain_type: Optional[str]
    metadata_path: Optional[str]
    answer: str
    sources: List[AggregatedToolSources]
    feedback: Optional[Feedback]
    timestamp: float
    llm_context: Optional[LLMContext]
    llm_kb_selection: Any
    llm_db_selection: Any
    generated_media: Optional[Dict[str, List[GeneratedMedia]]]
    generated_images: Optional[List[GeneratedMedia]]
    uploaded_docs: Optional[List[MediaSummary]]
    uploaded_image: Optional[str]  # TODO legacy to remove


class Conversation(TypedDict):
    id: str
    name: str
    timestamp: float
    auth_identifier: str
    data: List[QuestionData]
    media_summaries: Optional[List[MediaSummary]]
    conversation_type: str


class ExtractedQueryInfo(TypedDict, total=False):
    query: str
    query_index: int
    filters: Optional[Dict[str, List[Any]]]
    chain_type: Optional[str]
    media_summaries: Optional[List[MediaSummary]]
    previous_media_summaries: Optional[List[MediaSummary]]
    conversation_name: Optional[str]
    history: List
    answer: str
    sources: List[AggregatedToolSources]
    conversation_id: Optional[str]
    is_new_conversation: Optional[bool] # 'Agent Connect' specific for now
    knowledge_bank_id: Optional[str]
    retrieval_enabled: bool
    retrieval_selection: Optional[Dict[str, List[Any]]]
    llm_context: Optional[LLMContext]
    user_profile: Optional[Dict[str, Any]]
    generated_images: Optional[List[GeneratedMedia]]
    conversation_type: ConversationType
    app_id: Optional[str]


class UploadChainTypes(Enum):
    IMAGE = "image"
    DOCUMENT_AS_IMAGE = "document_as_image"
    SHORT_DOCUMENT = "short_document"
    LONG_DOCUMENT = "long_document"


class UploadFileError(Enum):
    GENERIC_ERROR = "generic_upload_error"
    NO_SELECTED_FILE = "no_selected_file"
    CONTEXT_EXCURSION = "context_excursion"
    INVALID_FILE_TYPE = "invalid_file_type"
    FILE_TOO_LARGE = "file_too_large"
    PARSING_ERROR = "parsing_error"
    TOO_MANY_FILES = "too_many_files"


class UploadFileResponse(TypedDict):
    media_summaries: List[MediaSummary]


class RecordState(Enum):
    PRESENT = "present"
    DELETED = "deleted"
    CLEARED = "cleared"
    EDITED = "edited"


class ConversationInsertInfo(TypedDict, total=False):
    name: Optional[str]
    platform: Optional[str]
    metadata: Optional[Dict[str, Any]]


class MessageInsertInfo(TypedDict, total=False):
    platform: str
    llm_name: Optional[str]
    query: str
    answer: Optional[str]
    filters: Optional[Dict[str, List[Any]]]
    sources: Optional[Union[List[Source], List[Any]]]
    llm_context: Optional[LLMContext]
    generated_media: Optional[Dict[str, List[GeneratedMedia]]]
    feedback: Optional[Dict]
    conversation_id: Optional[str]
    history: Optional[List[LlmHistory]]


# Conversations endpoint objects, built after their marsmallow schemas

class APIFeedback(TypedDict, total=False):
    value: Union[FeedbackValue, str]
    choice: Optional[List[str]]
    message: Optional[str]

class APIProcessedFile(TypedDict, total=False):
    name : str
    format : str
    path : str
    thumbnail : Optional[str]
    chainType : Optional[str]
    jsonFilePath : Optional[str]


class APIRetrieval(TypedDict, total=False):
    name : str
    type : RetrieverMode
    alias : Optional[str]
    filters : Optional[Dict[str, List[Any]]]
    sources : Optional[Union[List[Source], List[Any]]]
    generatedSqlQuery : Optional[str]
    usedTables : Optional[List[str]]

class APIGeneratedMedia(TypedDict):
    data : str
    format : str
    path : str
    referredFilePath : str


class APIMessageResponse(TypedDict, total=False):
    id : str
    createdAt : str
    query : str
    answer : str
    usedRetrieval : APIRetrieval 
    feedback : Optional[APIFeedback]
    files : Optional[List[APIProcessedFile]]
    generatedMedia : List[APIGeneratedMedia]


class APIApplicationContext(TypedDict):
    applicationId : str


class APIConversationTitle(TypedDict, total=False):
    original : str
    edited : Optional[str]
    createdAt : Optional[str]


class APIConversationMetadata(TypedDict):
    id : str
    title : Optional[APIConversationTitle]
    createdAt : str
    lastMessageAt : str
    state : str


class APIConversationsResponse(TypedDict):
    user : str
    context : APIApplicationContext
    conversations : List[APIConversationMetadata]


class APISingleConversationResponse(TypedDict):
    user : str
    id : str
    context : APIApplicationContext
    title : Optional[APIConversationTitle]
    messages : List[APIMessageResponse]
    createdAt : str
    lastMessageAt : str
    state : str