import base64
from io import BytesIO
from typing import Optional

from common.backend.models.base import MediaSummary, UploadChainTypes, UploadFileError
from common.backend.utils.picture_utils import resize_with_ratio
from common.backend.utils.sql_timing import log_query_time
from common.backend.utils.upload_utils import save_extracted_json
from common.llm_assist.logging import logger
from common.solutions.chains.summary.image_summary_chain import ImageSummaryChain
from PIL import Image


@log_query_time
def extract_image_summary(
    file_path: str, file_data: bytes, original_file_name: str, language: Optional[str], begin_time: int
) -> MediaSummary:
    summary: Optional[MediaSummary] = None
    error_message: Optional[str] = None
    try:
        image = Image.open(BytesIO(file_data))
        resized_image = resize_with_ratio(image)
        buffered = BytesIO()
        resized_image.save(buffered, format="PNG")
        b64_image = base64.b64encode(buffered.getvalue()).decode("utf-8")
        preview = f"data:image/png;base64,{b64_image}"
        chain_type = UploadChainTypes.IMAGE
        summary = ImageSummaryChain(b64_image, original_file_name, language).get_summary()
        if summary is None:
            raise Exception(UploadFileError.PARSING_ERROR.value)
        summary["preview"] = preview
        summary["begin_time"] = begin_time
        summary = MediaSummary(**summary)
        metadata_path: str = save_extracted_json(file_path, summary)
        summary = {**summary, "file_path": file_path, "metadata_path": metadata_path, "chain_type": chain_type.value}
        return summary
    except Exception as e:
        logger.exception(f"Error during uploaded image parsing: {e}")
        if error_message:
            raise e
        raise Exception(UploadFileError.PARSING_ERROR.value)
