from typing import Any, Dict, Optional

from common.backend.utils.context_utils import LLMStepName
from common.solutions.chains.generic_decision_chain import GenericDecisionJSONChain
from dataiku.langchain.dku_llm import DKULLM


class ImageGenerationDecisionChain(GenericDecisionJSONChain):
    def __init__(self, llm: DKULLM, system_prompt: Optional[str]) -> None:
        self._llm = llm
        system_prompt = system_prompt or ""
        examples = """
            ## Example 1:
            [{"role": "user", "content":"generate blue circle"},
            {"role": "assistant", "content":'{"generated_media_by_ai": {"images": [{"file_path": "admin_1724677117_04RMXS.png", "file_format": "png", "referred_file_path":""}]}}'},
            {"role": "user", "content":"What are the revenues for Q1 2023 and Q2 in 2023?"}]
            Expected Answer: {"query": null, "justification": "Not an image generation query", "referred_image": null}

            ## Example 2:
            [{"role": "user", "content":"How are you today?"}]
            Expected Answer: {"query": null, "justification": "Not an image generation query", "referred_image": null}

            ## Example 3:
            [{"role": "user", "content":"hello"},
            {"role": "assistant", "content":"How can I help you today?"},
            {"role": "user", "content": "Generate an image with a red circle and a blue square."},]
            Expected Answer: {"query": "Generate an image with a red circle and a blue square", "justification": "Image generation query", "referred_image": null}

            ## Example 4:
            [{"role": "user", "content":"generate blue circle"},
            {"role": "assistant", "content":'{"generated_media_by_ai": {"images": [{"file_path":"admin_1724677117_04RMXS.png", "file_format": "png", "referred_file_path":""}]}}'}],
            {"role": "user", "content":"hello"},
            {"role": "assistant", "content":"Error when attempting to call model or agent."},
            {"role": "user", "content":"change color to red"},
            {"role": "assistant", "content":'{"generated_media_by_ai": {"images": [{"file_path":"admin_1724677570_LFUUQS.png", "file_format": "png", "referred_file_path":"admin_1724677117_04RMXS"}]}}'}],
            {"role": "user", "content":"change color to yellow"},
            {"role": "assistant", "content":'{"generated_media_by_ai": {"images": [{"file_path":"admin_1724677825_B7EGNE.png", "file_format": "png", "referred_file_path":"admin_1724677570_LFUUQS"}]}}'}],
            {"role": "user", "content":"change to red again"},]
            Expected Answer: {"query": "Generate an image with a red circle", "justification": "Image generation query referring to a previous image in the chat", "referred_image": "admin_1724677825_B7EGNE.png"}
            """

        self._prompt = (
            "Examine the user query and the chat history to determine if an image generation is needed. "
            "Identify if it is a request to generate an image and craft a new query for the image generation agent to perform at best the image generation task. "
            "Image generation agent doesn't have access to chat history. It can only generate images based on the query you generate and an image as reference when needed.\n"
            f"{system_prompt}\n"
            "# INSTRUCTIONS\n"
            "1 - Return only a JSON object with three keys: 'query', 'justification' and 'referred_image'. If no query is needed, 'query' should be null and 'justification' should explain why no query is necessary, same for 'referred_image'.\n"
            "   - 'query': The exact query to generate the desired image, or null if image generation is not needed.\n"
            "   - 'justification': A brief explanation detailing why the query is needed or not.\n"
            "   - 'referred_image': The image's file path that the user is referring to, or null if no image is referred. File path of previously generated images can be found in the chat history in generated_media_by_ai > images > file_path. \n"
            "2 - Provide only the JSON object. Do not include any other explanations, decorations, or additional information.\n"
            "3 - Ensure the JSON object is in a format that can be parsed directly by a JSON parser without needing any preprocessing. Use double quotes for keys and string values.\n"
            "4 - If multiple images could be referenced, choose the most relevant or the most recent one based on the user's query context. The most recent one in the chat history is the closest one to the user query.\n"
            '5 - If no image generation is needed, answer with JSON object: {"query": null, "justification": "your justification for no need for image generation", "referred_image": null}.\n'
            "6 - If the user asks the same query multiple times, you should generate a new image generation query each time without referencing previous images.\n"
            "# EXAMPLES:\n"
            f"{examples}\n"
            "# END OF EXAMPLES\n"
        )

    @property
    def prompt(self):
        return self._prompt

    @property
    def llm(self) -> DKULLM:
        return self._llm

    @property
    def chain_purpose(self) -> str:
        return LLMStepName.DECISION_IMAGE_GENERATION.value

    def verified_json_output(self, json_output) -> Dict[str, Any]:
        if "query" not in json_output or not json_output["query"]:
            return {
                "decision": False,
                "query": None,
                "justification": json_output.get("justification"),
                "referred_image": None,
            }

        return {
            "decision": True,
            "query": json_output.get("query"),
            "justification": json_output.get("justification"),
            "referred_image": json_output.get("referred_image"),
        }

