from typing import Any, Dict, List, Optional

from answers.solutions.knowledge_bank import get_core_knowledge_bank
from common.backend.utils.dataiku_api import dataiku_api
from dataiku.core.knowledge_bank import MultipartContext
from langchain.schema.document import Document


def from_documents_to_multipart_context(docs: List[Document], kb_id: str) -> MultipartContext:
    kb = get_core_knowledge_bank(kb_id)
    retrieved_context = MultipartContext()
    for doc in docs:
        doc.metadata["parts"] = []
        mpc = kb.get_multipart_context([doc])
        for part in mpc.parts:
            if part.type == "IMAGE_INLINE":
                doc.metadata["parts"].append(part)
        retrieved_context.parts.extend(mpc.parts)
    return retrieved_context


def from_documents_to_structured_context(
    list_of_documents: List[Document], kb_metadata_schema: Optional[List[Dict[str, Any]]]
) -> List[Dict[str, Any]]:
    structured_documents = []
    metadata_fields = dataiku_api.webapp_config.get("knowledge_sources_context_metadata")
    for document_index, document in enumerate(list_of_documents):
        if kb_metadata_schema and metadata_fields:
            filtered_metadata = [
                {
                    "description": schema.get("comment", ""),
                    "name": schema.get("name", ""),
                    "value": document.metadata[schema.get("name", "")],
                }
                for schema in kb_metadata_schema
                if schema.get("name", "") in document.metadata and schema.get("name", "") in metadata_fields
            ]

            structured_document = {
                "Source number": document_index + 1,
                "Content": document.page_content,
                "Metadata": filtered_metadata,
            }
        else:
            structured_document = {
                "Source number": document_index + 1,
                "Content": document.page_content,
            }
        structured_documents.append(structured_document)
    return structured_documents
