from datetime import datetime
from typing import Any, Dict, List, Optional, Tuple, Union

from common.backend.constants import PROMPT_DATE_FORMAT
from common.backend.models.base import (
    ConversationParams,
    LLMContext,
    LlmHistory,
    LLMStep,
    MediaSummary,
    RetrievalSummaryJson,
)
from common.backend.models.source import AggregatedToolSources
from common.backend.utils.context_utils import LLMStepName
from common.backend.utils.llm_utils import handle_prompt_media_explanation
from common.backend.utils.prompt_utils import append_user_profile_to_prompt
from common.llm_assist.logging import logger
from common.solutions.chains.generic_answers_chain import GenericAnswersChain
from dataiku.core.knowledge_bank import MultipartContext
from dataiku.langchain.dku_llm import DKULLM


class NoRetrievalChain(GenericAnswersChain):
    def __init__(self, llm: DKULLM, chat_has_media: bool = False, include_user_profile_in_prompt: Optional[bool]=False):
        super().__init__()
        self.__llm = llm
        self.__act_like_prompt = ""
        self.__system_prompt = ""
        self.__prompt_with_media_explanation = chat_has_media
        self.include_user_profile_in_prompt = include_user_profile_in_prompt

    @property
    def act_like_prompt(self) -> str:
        return self.__act_like_prompt

    @property
    def system_prompt(self) -> str:
        return self.__system_prompt

    @property
    def llm(self) -> DKULLM:
        return self.__llm

    @property
    def chain_purpose(self) -> str:
        return LLMStepName.NO_RETRIEVAL.value

    def load_role_and_guidelines_prompts(self, params: ConversationParams):
        act_like_prompt, system_prompt = self.load_default_role_and_guidelines_prompts()
        user_profile = params.get("user_profile", None)
        system_prompt = append_user_profile_to_prompt(
            system_prompt=system_prompt, user_profile=user_profile, include_full_user_profile=self.include_user_profile_in_prompt
        )
        system_prompt = handle_prompt_media_explanation(
            system_prompt=system_prompt, has_media=self.__prompt_with_media_explanation
        )
        self.__act_like_prompt = act_like_prompt
        self.__system_prompt = system_prompt


    def get_computed_system_prompt(self, params: ConversationParams) -> str:
        datetime_now = datetime.now().strftime(PROMPT_DATE_FORMAT)
        return rf"""
        Today's date and time: {datetime_now}
        {self.act_like_prompt}
        
        {self.system_prompt}

        """

    def get_computing_prompt_step(self) -> LLMStep:
        return LLMStep.COMPUTING_PROMPT_WITHOUT_RETRIEVAL

    def get_querying_step(self, params: ConversationParams) -> LLMStep:
        return LLMStep.QUERYING_LLM_WITHOUT_RETRIEVAL

    def finalize_non_streaming(
        self,
        params: ConversationParams,
        answer_context: Union[str, Dict[str, Any], List[str]],
    ) -> RetrievalSummaryJson:
        return self.get_as_json(
            generated_answer=answer_context,
            uploaded_docs=params.get("media_summaries", None),
            sources=answer_context.get("aggregated_sources_list", None) # type: ignore
        )

    def finalize_streaming(
        self,
        params: ConversationParams,
        answer_context: Union[str, Dict[str, Any], List[str]],
    ) -> RetrievalSummaryJson:
        return self.get_as_json(
            generated_answer=answer_context, 
            uploaded_docs=params.get("media_summaries", None),
            sources=answer_context.get("aggregated_sources_list", None) # type: ignore
        )


    def get_retrieval_context(self, params: ConversationParams,
    ) -> Tuple[Optional[MultipartContext], Dict[str, Any]]:
        return None, {}

    def get_as_json(
        self,
        generated_answer: Union[str, Dict[str, Any], List[str]],
        user_profile: Optional[Dict[str, Any]] = None,
        uploaded_docs: Optional[List[MediaSummary]] = None,
        sources: Optional[List[AggregatedToolSources]] = [],
    ) -> RetrievalSummaryJson:
        llm_context: LLMContext = {}  # type: ignore
        if user_profile:
            llm_context["user_profile"] = user_profile
        if uploaded_docs:
            logger.debug(f"uploaded_docs:  {uploaded_docs}", log_conv_id=True)
            llm_context["uploaded_docs"] = [
                {
                    "original_file_name": str(uploaded_doc.get("original_file_name")),
                    "file_path": str(uploaded_doc.get("file_path")),
                    "metadata_path": str(uploaded_doc.get("metadata_path")),
                }
                for uploaded_doc in uploaded_docs
            ]
        if isinstance(generated_answer, str):
            return RetrievalSummaryJson(
                answer=generated_answer,
                sources=sources if sources else [],
                filters=None,
                knowledge_bank_selection=[],
                llm_context=llm_context,
                user_profile=user_profile if user_profile else {},
            )

        if isinstance(generated_answer, dict):
            answer = generated_answer.get("answer", "")
            if generated_answer.get("images", None):
                return RetrievalSummaryJson(
                    answer=answer,
                    generated_images=generated_answer.get("images", []),
                    sources=sources if sources else [],
                    filters=None,
                    llm_context=llm_context,
                    user_profile=generated_answer.get("user_profile", user_profile),
                )
            return RetrievalSummaryJson(
                answer=answer, sources=sources if sources else [], filters=None, llm_context=llm_context
            )
        logger.error(f"Generated answer type not supported. This should not happen. {generated_answer}")
        return {}

    def create_query_from_history_and_update_params(
        self, chat_history: List[LlmHistory], user_query: str, conversation_params: ConversationParams
    ) -> ConversationParams:
        return conversation_params
