import dataiku
from dataiku.customrecipe import (
    get_input_names_for_role,
    get_output_names_for_role,
    get_recipe_config
)
from dataset_name_search import DatasetNameSearch
from workbook import (
    read_workbook_from_managed_folder,
    populate_workbook_from_dataset,
    write_workbook_to_managed_folder
)
from helper import check_file_name, CheckFileName
import logging

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO, format='Excel Templater | %(levelname)s - %(message)s')

# Checks the output file name
output_file_name = get_recipe_config().get('output_file_name')
check = check_file_name(output_file_name) 
if check == CheckFileName.EMPTY:
    raise AttributeError("The output file name cannot be empty")
elif check == CheckFileName.WITH_EXTENSION:
    raise AttributeError("The output file name should not already contain an extension")

# Get handles in INPUT and OUTPUT
# Get handle on input folders
input_folder_name = get_input_names_for_role('input_folder')[0]
input_folder = dataiku.Folder(input_folder_name)

# Checks the template file
selected_template_file_name = get_recipe_config().get("template_file", None)
if selected_template_file_name is None:
    raise AttributeError("A template file should be selected")

current_project_key = dataiku.default_project_key()
if not isinstance(current_project_key, str):
    raise RuntimeError("Could not retrieve the current project key")

input_dataset_names = get_input_names_for_role('input_datasets')
logger.info(f"Input datasets: {input_dataset_names}")

dataset_name_search = DatasetNameSearch(input_dataset_names, current_project_key)

# Get handle on output folder
output_folder_name = get_output_names_for_role('output_folder')[0]
output_folder = dataiku.Folder(output_folder_name)

# Retrieve mandatory user-defined parameters
dataset_keyword = get_recipe_config().get('dataset_keyword', "DATASET")
if not dataset_keyword:
    dataset_keyword = "DATASET"

# Checks if the user wants to output the column names
include_column_headers = get_recipe_config().get("include_column_headers")
# Handles backward compatibility
if include_column_headers is None:
    include_column_headers = False

logger.info(
    "Starting Excel Templater plugin with keyword={}, file name = {}, input folder = {}, include column headers = {}".format(
        dataset_keyword, output_file_name, input_folder_name, include_column_headers
    )
)

workbook = read_workbook_from_managed_folder(input_folder, selected_template_file_name)
populate_workbook_from_dataset(workbook, dataset_keyword, dataset_name_search, include_column_headers)
write_workbook_to_managed_folder(workbook, output_folder, output_file_name + ".xlsx")
