import logging
import math
import re
from typing import Tuple

logger = logging.getLogger(__name__)
logging.basicConfig(
    level=logging.INFO, format="Excel Templater | %(levelname)s - %(message)s"
)


def parse_cell_range(cell_range) -> Tuple[int, int, int, int]:
    """
    Parses a string representation of a cell range like "A15:E42" to 1, 15, 5, 42
    """
    start_cell, end_cell = cell_range.split(":")
    start_column, start_row = parse_cell(start_cell)
    end_column, end_row = parse_cell(end_cell)
    return start_column, start_row, end_column, end_row


def parse_cell(cell_str) -> Tuple[int, int]:
    """
    Parses a string representation of a cell like "E42" to 5, 42
    """
    match = re.search(r"([A-Za-z]+)(\d+)", cell_str)
    if match is None or len(match.groups()) != 2:
        raise ValueError(
            f"Could not parse cell reference {cell_str} into column and row"
        )

    column_str: str = match.group(1)
    row_str: str = match.group(2)

    row = int(row_str)
    column = 0
    for i, letter in enumerate(reversed(column_str)):
        column += math.pow(26, i) * (ord(letter.upper()) - ord("A") + 1)

    return int(column), row


def cell_range_to_string(
    start_column: int, start_row: int, end_column: int, end_row: int
) -> str:
    """
    Returns a string representation of a cell, from 1, 15, 5, 42 to "A15:E42"
    """
    return f"{cell_to_string(start_column, start_row)}:{cell_to_string(end_column, end_row)}"


def cell_to_string(column: int, row: int) -> str:
    """
    Returns a string representation of a cell range, from 5, 42 to "E42"
    """
    column_str = ""
    while column > 0:
        # Computes the letter representation going from the lowest digit position to highest
        current_digit = column % 26  # In base-26
        current_letter = chr(current_digit - 1 + ord("A"))
        column_str = f"{current_letter}{column_str}"
        column = column // 26

    return f"{column_str}{row}"
