from typing import List, Optional
from typing import NamedTuple

import logging

logger = logging.getLogger(__name__)
logging.basicConfig(
    level=logging.INFO, format="Excel Templater | %(levelname)s - %(message)s"
)


class DatasetFullyQualifiedName(NamedTuple):
    project_key: str
    dataset_name: str
    
    def to_string(self):
        return f"{self.project_key}.{self.dataset_name}"


class DatasetNameSearch(object):
    def __init__(self, dataset_full_names: List[str], current_project_key: str):
        """
        Search service on datasets name and associated project key (fully qualified name)

        :param dataset_full_names: List of dataset fully qualified names as string
        :param current_project_key: Project key if the current project
        """
        # Separated into current project and shared datasets to be able to prioritize current project datasets when searching
        self.current_project_datasets: List[DatasetFullyQualifiedName] = []
        self.shared_datasets: List[DatasetFullyQualifiedName] = []
        self.current_project_key = current_project_key

        for dataset_full_name in dataset_full_names:
            [project_key, dataset_name] = dataset_full_name.split(".")
            if project_key is None or dataset_name is None:
                logger.warning(f"Dataset {dataset_full_name} could not be parsed")
                continue

            dataset_info = DatasetFullyQualifiedName(
                project_key=project_key, dataset_name=dataset_name
            )
            if project_key == current_project_key:
                self.current_project_datasets.append(dataset_info)
            else:
                self.shared_datasets.append(dataset_info)

    def find_from_tag(self, tag_value: str) -> Optional[DatasetFullyQualifiedName]:
        """
        Finds the associated existing dataset name (with its project key) matching a tag value

        Performs a case insensitive search on the current project datasets and the shared datasets.
        The current project datasets are prioritized in the search (when there is no project key in the `tag_value`).

        :param tag_value: Represents the wanted dataset. This method handles the following format:
        - `<tag>.<dataset_name>`
        - `<tag>.<project_key>.<dataset_name>`

        :return: a dataset fully-qualified name tuple containing the dataset name and associated project key
        """
        splitted_tag = tag_value.split(".")

        # Only tries to match on the dataset name, case insensitively
        if len(splitted_tag) == 2:
            input_name = splitted_tag[1]

            # First from current project datasets
            for dataset_info in self.current_project_datasets:
                if input_name.lower() == dataset_info.dataset_name.lower():
                    return dataset_info

            # Otherwise from shared datasets
            for dataset_info in self.shared_datasets:
                if input_name.lower() == dataset_info.dataset_name.lower():
                    return dataset_info

        # Tries to match the project key and dataset name, case insensitively
        elif len(splitted_tag) == 3:
            input_project_key = splitted_tag[1]
            input_name = splitted_tag[2]

            if input_project_key == self.current_project_key:
                for dataset_info in self.current_project_datasets:
                    if (
                        input_project_key.lower() == dataset_info.project_key.lower()
                        and input_name.lower() == dataset_info.dataset_name.lower()
                    ):
                        return dataset_info

            else:
                # Otherwise from shared datasets
                for dataset_info in self.shared_datasets:
                    if (
                        input_project_key.lower() == dataset_info.project_key.lower()
                        and input_name.lower() == dataset_info.dataset_name.lower()
                    ):
                        return dataset_info

        return None
