from geo_router import GeoRouter
from dkuconstants import DISTANCE_UNIT
from dkuconstants import CONVERSION
import utils


class GraphhopperGeoRouter(GeoRouter):
    def __init__(self, api_endpoint_url, dku_license_id=None):
        super(GraphhopperGeoRouter, self).__init__(api_endpoint_url, dku_license_id)

    def enrich_headers(self, headers):
        super(GraphhopperGeoRouter, self).enrich_headers(headers)
        headers["X-Prefered-Engine"] = "Graphhopper"
        headers["X-Api-Version"] = "None"

    @staticmethod
    def get_isochrone_geometry_from_response(response):
        return response.get("polygons")

    @staticmethod
    def get_itinerary_from_response(response):
        itinerary = utils.decode_encoded_polyline(response["paths"][0]["points"])
        if len(itinerary) <=1:
            return {"type":"LineString", "coordinates": []}
        return {"type": "LineString", "coordinates": itinerary}

    @staticmethod
    def get_time_from_response(response):
        time_in_ms = response["paths"][0]["time"]
        return round(time_in_ms / CONVERSION.MINUTE_IN_MILLISECONDS.value)

    @staticmethod
    def get_distance_from_response(response, distance_unit):
        distance_in_meters = response["paths"][0]["distance"]
        if distance_unit == DISTANCE_UNIT.METERS:
            return distance_in_meters
        elif distance_unit == DISTANCE_UNIT.MILES:
            return distance_in_meters / CONVERSION.MILE_IN_METERS.value
        elif distance_unit == DISTANCE_UNIT.KILOMETERS:
            return distance_in_meters / CONVERSION.KILOMETER_IN_METERS.value

    @staticmethod
    def get_isochrone_querystring(coords, time_threshold, global_transport_mode):
        return {
            "point": f"{coords[1]},{coords[0]}",
            "time_limit": int(time_threshold * 60),
            "profile": utils.get_local_transport_mode(global_transport_mode),
        }

    @staticmethod
    def get_route_querystring(from_coords, to_coords, global_transport_mode, get_itinerary):
        return {
            "point": [f"{from_coords[1]},{from_coords[0]}", f"{to_coords[1]},{to_coords[0]}"],
            "profile": utils.get_local_transport_mode(global_transport_mode),
            "points_encoded": True,
            "calc_points": get_itinerary,
            "instructions" : False
        }