"""This module implements a client to request Overpass API, using api-client package.
See https://github.com/MikeWooster/api-client/blob/master/README.md for more details"""

from apiclient import APIClient, endpoint, retry_request
from apiclient.retrying import retry_if_api_request_error, retry_request
from apiclient.response import RequestsResponse
from apiclient.error_handlers import BaseErrorHandler
from apiclient import exceptions
from apiclient.response import Response
import tenacity
from apiclient.exceptions import UnexpectedError

# rewrite method from api-client to get a clearer error message
def custom_make_request(
    self,
    request_method,
    endpoint: str,
    params=None,
    headers=None,
    data=None,
    **kwargs,
) -> Response:
    """Make the request with the given method.
    Delegates response parsing to the response handler.
    """
    try:
        response = RequestsResponse(
            request_method(
                endpoint,
                params=self._get_request_params(params),
                headers=self._get_request_headers(headers),
                auth=self._get_username_password_authentication(),
                data=self._get_formatted_data(data),
                timeout=2000, #self._get_request_timeout(),
                **kwargs,
            )
        )
    except Exception as error:
        raise UnexpectedError(f"Error when contacting '{endpoint}' with error {error}") from error
    else:
        self._check_response(response)
    return self._decode_response_data(response)


# class to define osm url endpoints
@endpoint(base_url="http://overpass-api.de")
class Endpoint:
    interpreter = "api/interpreter"


# class to define a retrying strategy
class RetryDecorator:
    def retry_for_specific_errors(self, errors, wait, stop, reraise):
        return tenacity.retry(
            retry=retry_if_api_request_error(status_codes=errors),
            wait=tenacity.wait_fixed(wait),
            stop=tenacity.stop_after_attempt(stop),
            reraise=reraise,
        )


# class to define the client for OSM
class OSMClient(APIClient):
    retry = RetryDecorator().retry_for_specific_errors([503, 504, 429, 500, 502, 522], 18, 3, True)

    @retry
    def post_pois(self, request):
        return self.post(Endpoint.interpreter, request)


# class to define a strategy for API errors
class CustomErrorHandler(BaseErrorHandler):
    @staticmethod
    def get_exception(response: Response) -> exceptions.APIRequestError:
        status_code = response.get_status_code()
        exception_class = exceptions.UnexpectedError
        if 300 <= status_code < 400:
            exception_class = exceptions.RedirectionError
        elif 400 <= status_code < 500:
            exception_class = exceptions.ClientError
        elif 500 <= status_code < 600:
            exception_class = exceptions.ServerError
        return exception_class(
            message=(
                f"{status_code} Error: {response.get_status_reason()} "
                f"for url: {response.get_requested_url()}"
                f"with response: {response.get_raw_data()}"
            )
        )