# -*- coding: utf-8 -*-
import logging
import requests
from time import time

import dataiku
from dataiku.runnables import Runnable


class MyRunnable(Runnable):
    """The base interface for a Python runnable"""

    MODEL_URL = "https://github.com/mozilla/DeepSpeech/releases/download/v0.9.1/deepspeech-0.9.1-models.pbmm"

    def __init__(self, project_key, config, plugin_config):
        """
        :param project_key: the project in which the runnable executes
        :param config: the dict of the configuration of the object
        :param plugin_config: contains the plugin settings
        """
        self.project_key = project_key
        self.config = config
        self.plugin_config = plugin_config
        self.use_ssl = self.plugin_config.get("use_ssl", True)
        if self.use_ssl:
            self.model_url = self.MODEL_URL
        else:
            self.model_url = self.MODEL_URL.replace("https://", "http://")
        self.client = dataiku.api_client()

    def get_progress_target(self):
        """
        If the runnable will return some progress info, have this function return a tuple of
        (target, unit) where unit is one of: SIZE, FILES, RECORDS, NONE
        """
        return (100, "NONE")

    def run(self, progress_callback):
        """
        Do stuff here. Can return a string or raise an exception.
        The progress_callback is a function expecting 1 value: current progress
        """
        output_folder_name = self.config["folder_name"]
        if not output_folder_name:
            raise ValueError("Please enter a folder name")

        # Creating new Managed Folder if needed
        project = self.client.get_project(self.project_key)
        for folder in project.list_managed_folders():
            if output_folder_name == folder["name"]:
                output_folder = project.get_managed_folder(folder["id"])
                break
        else:
            output_folder = project.create_managed_folder(output_folder_name)
        output_folder = dataiku.Folder(output_folder.get_definition()["id"], project_key=self.project_key)

        logging.info(f"Downloading DeepSpeech model at URL: {self.model_url} ...")
        start = time()
        with output_folder.get_writer("model.pbmm") as writer:
            with requests.get(self.model_url, stream=True, verify=self.use_ssl) as response:
                content_length = int(response.headers["content-length"])
                block_size = max(4096, content_length // 100)
                downloaded_size = 0
                for chunk in response.iter_content(block_size):
                    if not chunk:
                        break
                    writer.write(chunk)
                    downloaded_size += len(chunk)
                    progress_callback(int((downloaded_size / content_length) * 100))
        result_message = f"Downloading DeepSpeech model: Done in {time() - start:.2f} seconds."
        logging.info(result_message)
        return result_message
