"""Dynamic column choice filtering for the generate-synthetic-data recipe."""

from typing import List, Optional

import dataiku


TARGET_PARAMETERS = {"continuous_columns"}
INPUT_ROLE = "sensitive_dataset"


def _get_dataset(inputs: Optional[List[dict]]) -> Optional[dataiku.Dataset]:
    """Return the Dataset object for the sensitive input role if it is available."""
    if not inputs:
        return None
    for item in inputs:
        if item.get("role") != INPUT_ROLE:
            continue
        full_name = item.get("fullName") or ""
        dataset_name = item.get("datasetName")
        project_key = item.get("projectKey")
        if full_name:
            if "." in full_name:
                project_key, dataset_name = full_name.split(".", 1)
            else:
                dataset_name = full_name
        if not dataset_name:
            continue
        if project_key:
            return dataiku.Dataset(dataset_name, project_key=project_key)
        return dataiku.Dataset(dataset_name)
    return None


def _list_columns(dataset: Optional[dataiku.Dataset]) -> List[str]:
    if dataset is None:
        return []
    try:
        schema = dataset.read_schema()
    except Exception:
        return []
    return [col.get("name") for col in schema if col.get("name")]


def do(payload, config, plugin_config, inputs):
    current_config = config or {}
    parameter_name = (payload or {}).get("parameterName")
    if parameter_name not in TARGET_PARAMETERS:
        return {"choices": []}
    excluded_columns = set(current_config.get("excluded_columns") or [])
    continuous_selected = set(current_config.get("continuous_columns") or [])
    dataset = _get_dataset(inputs)
    base_columns = _list_columns(dataset)
    allowed_columns = [col for col in base_columns if col not in excluded_columns]
    # keep selected values even if they were filtered (for safety when columns disappear later)
    preserved = [col for col in continuous_selected if col not in allowed_columns]
    full_choices = allowed_columns + preserved
    seen = set()
    choices = []
    for col in full_choices:
        if col in seen:
            continue
        seen.add(col)
        choices.append({"value": col, "label": col})
    return {"choices": choices}
