import packageInfo from '../../package.json'
import { ViewMode } from './views'

const version = packageInfo.version

enum EventName {
  // traces management
  PasteNewTrace = 'trace-explorer-paste-new-trace',
  AddTrace = 'trace-explorer-add-trace',
  CancelPasteTrace = 'trace-explorer-cancel-paste-trace',
  ReloadTraces = 'trace-explorer-reload-traces',
  // tabs & view change
  TreeTabChange = 'trace-explorer-tree-tab-change',
  TimelineTabChange = 'trace-explorer-timeline-tab-change',
  ExplorerTabChange = 'trace-explorer-explorer-tab-change',
  ResetView = 'trace-explorer-reset-view',
  // explore traces & nodes
  SelectTrace = 'trace-explorer-select-trace',
  FilterTraces = 'trace-explorer-filter-trace',
  SearchNodes = 'trace-explorer-search-nodes',
  SortExplorerNodes = 'trace-explorer-sort-explorer-nodes',
  // expand & highlight nodes
  OpenNodeDetails = 'trace-explorer-open-node-details',
  HighlightNodes = 'trace-explorer-highlight-node',
  // error
  Error = 'trace-explorer-error' // error
}

export class WT1iser {
  private static webappVersion = version

  /* traces management */

  static pasteNewTrace() {
    this.sendEvent(EventName.PasteNewTrace, {})
  }

  static addTrace() {
    this.sendEvent(EventName.AddTrace, {})
  }

  static cancelPasteTrace() {
    this.sendEvent(EventName.CancelPasteTrace, {})
  }

  static reloadTraces() {
    this.sendEvent(EventName.ReloadTraces, {})
  }

  /* tabs & view change */

  static tabChange(tab: ViewMode) {
    if (tab === ViewMode.Tree) {
      this.sendEvent(EventName.TreeTabChange, {})
    } else if (tab === ViewMode.Timeline) {
      this.sendEvent(EventName.TimelineTabChange, {})
    } else if (tab === ViewMode.Explorer) {
      this.sendEvent(EventName.ExplorerTabChange, {})
    }
  }

  static treeTabChange() {
    this.sendEvent(EventName.TreeTabChange, {})
  }

  static timelineTabChange() {
    this.sendEvent(EventName.TimelineTabChange, {})
  }

  static explorerTabChange() {
    this.sendEvent(EventName.ExplorerTabChange, {})
  }

  static resetView(viewMode: ViewMode) {
    this.sendEvent(EventName.ResetView, { viewMode: viewMode })
  }

  /* explore traces & nodes */

  static selectTrace(traceId: string) {
    this.sendEvent(EventName.SelectTrace, { traceId })
  }

  static filterTraces(nameFilter: string, durationRange: string, dateFrom: string, dateTo: string) {
    this.sendEvent(EventName.FilterTraces, {
      nameFilter,
      durationRange,
      dateFrom,
      dateTo
    })
  }

  static searchNodes(query: string) {
    this.sendEvent(EventName.SearchNodes, { query })
  }

  static sortExplorerNodes(sorters: any) {
    this.sendEvent(EventName.SortExplorerNodes, { sorters })
  }

  static openNodeDetails(nodeId: string) {
    this.sendEvent(EventName.OpenNodeDetails, { nodeId })
  }

  static highlightNodes(group: string) {
    this.sendEvent(EventName.HighlightNodes, { group })
  }

  /* error */

  static error(actionName: string, nQuestions: number, error: string): void {
    this.sendEvent(EventName.Error, { actionName, nQuestions, error })
  }

  static init() {
    console.log('** trace explorer version **', version)
    try {
      if (!(window.parent as any).WT1SVC && (window as any).dkuUsageReportingUtils) {
        console.debug('bootstrap standalone reporting mode')
        ;(window as any).dkuUsageReportingUtils.standaloneModeBootstrap()
      }
    } catch (error) {
      console.error('Error in WT1iser.init', error)
    }
  }

  private static prepareEventParams(props: any) {
    return {
      ...props,
      webappVersion: this.webappVersion
    }
  }

  private static sendEvent(event: EventName, props: any) {
    try {
      const params = this.prepareEventParams(props)
      if ((window.parent as any).WT1SVC) {
        ;(window.parent as any).WT1SVC.event(event, params)
      } else if ((window as any).dkuUsageReportingUtils) {
        ;(window as any).dkuUsageReportingUtils.standaloneModeTrackEvent(event, params)
      }
    } catch (error) {
      console.error('Error in WT1iser.sendEvent', error)
    }
  }
}
