export interface Trace {
  id: string
  parentNode?: TraceNode
  name?: string
  result?: string
  begin?: string
  duration?: number
  date?: Date
  tokens?: UsageMetadata
}

export interface TraceNode {
  id?: string
  traceName?: string
  type: string
  begin?: string
  end?: string
  timestamp?: string
  date?: Date
  duration?: number
  name: string
  children: (TraceNode | TraceNodeEvent)[]
  attributes: SpanAttributes
  usageMetadata?: UsageMetadata
  inputs?: SpanInputsOutputs
  outputs?: SpanInputsOutputs
}

export interface TraceNodeEvent {
  type: string
  timestamp: string
  name: string
  children: any[]
  attributes: Record<string, any>
  id?: string
}

export interface UsageMetadata {
  promptTokens: number
  completionTokens: number
  totalTokens: number
  estimatedCost: number
  tokenCountsAreEstimated?: boolean
}

// -----------------------------------------------------------------------------
// Message Parts (multipart messages with images, etc.)
// -----------------------------------------------------------------------------

export interface MessagePart {
  type: string
  text?: string
  mimeType?: string
  dataBase64?: string
  uri?: string
}

// -----------------------------------------------------------------------------
// Tool Calls & Outputs (embedded in messages)
// -----------------------------------------------------------------------------

/** Tool call embedded in assistant message */
export interface MessageToolCall {
  type: string
  function: {
    name: string
    arguments: string
  }
  id: string
  index: number
  llmReasoning?: unknown
}

/** Tool output embedded in tool message */
export interface MessageToolOutput {
  callId: string
  output: string
  parts?: MessagePart[]
}

/** Merged tool call with its output (for display) */
export interface MergedToolCall {
  callId: string
  name: string
  arguments: string
  output?: string
}

// -----------------------------------------------------------------------------
// Messages
// -----------------------------------------------------------------------------

export interface Message {
  role: string
  content?: string
  text?: string
  parts?: MessagePart[]
  toolCalls?: MessageToolCall[]
  toolOutputs?: MessageToolOutput[]
  memoryFragment?: unknown
  memoryFragmentTarget?: unknown
  toolValidationRequests?: unknown[]
  toolValidationResponses?: unknown[]
}

// -----------------------------------------------------------------------------
// Reranking Query
// -----------------------------------------------------------------------------

export interface RerankingQueryPart {
  type: string
  text?: string
}

export interface RerankingDocument {
  parts: RerankingQueryPart[]
}

export interface RerankingQuery {
  queryParts: RerankingQueryPart[]
  documents: RerankingDocument[]
}

// -----------------------------------------------------------------------------
// Sources & Artifacts (tool execution outputs)
// -----------------------------------------------------------------------------

export interface SourceItem {
  type?: string
  title?: string
  url?: string
  textSnippet?: string
  markdownSnippet?: string
  htmlSnippet?: string
  jsonSnippet?: string
  thumbnails?: unknown[]
  fileRef?: unknown
  imageRefs?: unknown[]
  filename?: string
  mimeType?: string
  dataBase64?: string
  columns?: string[]
  data?: unknown[][]
  text?: string
}

export interface Source {
  originToolId?: string
  originToolName?: string
  toolCallDescription?: string
  hierarchy?: unknown[]
  items?: SourceItem[]
}

export interface Artifact {
  id?: string
  type?: string
  name?: string
  hierarchy?: unknown[]
  parts?: unknown[]
}

// -----------------------------------------------------------------------------
// PII Detection
// -----------------------------------------------------------------------------

export interface DetectedEntity {
  type: string
}

// -----------------------------------------------------------------------------
// Span Attributes
// -----------------------------------------------------------------------------

interface CompletionQuery {
  messages: Message[]
}

interface CompletionQuerySettings {
  maxOutputTokens: number
  stopSequences: string[]
}

interface CompletionResponse {
  ok: boolean
  text: string
  finishReason: string
  toolCalls: any[]
  predictedClassProbas: any[]
  fromCache: boolean
  promptTokens?: number
  completionTokens?: number
  totalTokens?: number
  estimatedCost?: number
}

export interface SpanAttributes {
  llmId?: string
  llmProvider?: string
  llmModel?: string
  completionQuery?: CompletionQuery
  completionQuerySettings?: CompletionQuerySettings
  completionResponse?: CompletionResponse
  embeddingResponse?: unknown
  rerankingResponse?: unknown
}

// -----------------------------------------------------------------------------
// Span Inputs/Outputs (single type with all possible fields)
// -----------------------------------------------------------------------------

export interface SpanInputsOutputs {
  /** Completion query messages (inputs) */
  messages?: Message[]

  /** Embedding query text (inputs) / LLM response text (outputs) */
  text?: string

  /** Reranking query (inputs) */
  query?: RerankingQuery

  /** Tool execution input (inputs) - can be any JSON */
  input?: unknown

  /** PII guardrail checked text (inputs) */
  checkedText?: string

  /** LLM response tool calls (outputs) */
  toolCalls?: ToolCall[]

  /** LLM response tool validation requests (outputs, 14.4+) */
  toolValidationRequests?: unknown[]

  /** LLM response memory fragment (outputs, 14.4+) */
  memoryFragment?: unknown

  /** Tool execution output (outputs) - can be any JSON */
  output?: unknown

  /** Tool execution error (outputs) */
  error?: string

  /** Tool execution sources (outputs) */
  sources?: Source[]

  /** Tool execution artifacts (outputs) */
  artifacts?: Artifact[]

  /** PII detection results (outputs) */
  detectedEntities?: DetectedEntity[]
}

interface FunctionCall {
  arguments: string
  name?: string
}

export interface ToolCall {
  type: string
  function: FunctionCall
  index: number
  id?: string
}

export interface ChildListItem {
  name: string
  id?: string
  nodeGrp: string | null
  emptyNode?: boolean
}
