"""
Utility functions for Google Agent Engine integration with Dataiku.

This module provides shared authentication functionality for connecting to Google Vertex AI Agent Engine.
"""

import json
import logging
from google.oauth2 import service_account, credentials

# Create logger
logger = logging.getLogger(__name__)

# Required scopes for Vertex AI APIs (Langchain + ADK)
VERTEX_AI_SCOPES = [
    'https://www.googleapis.com/auth/cloud-platform'
]


def get_credentials_from_vertexai_connection(connection_info):
    """Get Google Cloud credentials from Dataiku Vertex AI connection.

    Args:
        connection_info: Connection info object from Dataiku API

    Returns:
        Google Cloud credentials object with required scopes for Vertex AI (Langchain + ADK)

    Raises:
        ValueError: If connection is invalid or missing required parameters
    """
    connection_params = connection_info.get_resolved_params()

    if connection_params['authType'] == "KEYPAIR":
        if 'appSecretContent' in connection_params:
            keyRaw = connection_params['appSecretContent']
        elif 'keyPath' in connection_params:
            keyRaw = connection_params['keyPath']
        else:
            raise ValueError(
                "No keypair found in connection. "
                "Please refer to DSS Service Account Auth documentation."
            )
        key = json.loads(keyRaw)
        
        # Create credentials with required scopes for Vertex AI APIs
        gcp_credentials = service_account.Credentials.from_service_account_info(
            key,
            scopes=VERTEX_AI_SCOPES
        )
        logger.info(f"Created service account credentials with scopes: {VERTEX_AI_SCOPES}")

    elif connection_params['authType'] == "OAUTH":
        if 'accessToken' not in connection_info['resolvedOAuth2Credential']:
            raise ValueError(
                "No accessToken found in connection. "
                "Please refer to DSS OAuth2 credentials documentation."
            )
        accessToken = connection_info['resolvedOAuth2Credential']['accessToken']
        gcp_credentials = credentials.Credentials(accessToken)
        logger.info("Created OAuth credentials with access token")

    else:
        raise ValueError(f"Unsupported authentication type '{connection_params['authType']}'.")

    return gcp_credentials
