import json
import logging
from google.oauth2 import service_account, credentials
from google.cloud.aiplatform_v1 import ReasoningEngineServiceClient
from google.cloud.aiplatform_v1.types import ListReasoningEnginesRequest

from utils import get_credentials_from_vertexai_connection

# Create logger
logger = logging.getLogger(__name__)


def do(payload, config, plugin_config, inputs):
    """
    Fetch the list of agents from Vertex AI Agent Engine

    Returns:
        Dictionary with choices list containing agent options
    """
    import dataiku

    # Get connection and parameters
    gcp_connection_name = config.get('vertexai_connection')
    agent_framework = config.get('agent_framework')
    logger.debug(f"GCP Connection Name: {gcp_connection_name}")

    # Validate required parameters
    if not gcp_connection_name:
        error_msg = "No Vertex AI connection specified. Please select a connection first."
        logger.error(error_msg)
        return {"choices": []}

    if not agent_framework:
        error_msg = "No Agent Framework specified. Please select an agent framework first."
        logger.error(error_msg)
        return {"choices": []}

    try:
        # Get GCP connection
        logger.debug(f"Fetching connection info for: {gcp_connection_name}")
        client = dataiku.api_client()
        connection = client.get_connection(gcp_connection_name)
        connection_info = connection.get_info()
        logger.info("Successfully retrieved connection info")
        # Get connection parameters
        conn_params = connection_info.get_params()
        gcp_project = conn_params.get('project')
        gcp_region = conn_params.get('region', 'us-central1')
        logger.info(f"GCP Project: {gcp_project}")
        logger.info(f"GCP Region: {gcp_region}")

        # Get credentials
        logger.info("Retrieving GCP credentials from connection")
        gcp_credentials = get_credentials_from_vertexai_connection(connection_info)
        logger.info("Successfully retrieved GCP credentials")

        # List agents from Vertex AI Agent Engine using Reasoning Engine Service
        # Using the parent path format: projects/{project}/locations/{location}
        parent = f"projects/{gcp_project}/locations/{gcp_region}"
        logger.debug(f"Listing agents from parent path: {parent}")

        # Fetch agents using the Reasoning Engine API
        agents_list = []
        try:
            # Create Reasoning Engine Service client
            logger.info("Creating ReasoningEngineServiceClient")
            reasoning_client = ReasoningEngineServiceClient(
                credentials=gcp_credentials,
                client_options={"api_endpoint": f"{gcp_region}-aiplatform.googleapis.com"})

            request = ListReasoningEnginesRequest(parent=parent)
            logger.info("Sending list reasoning engines request")

            # List all reasoning engines (agents) in the project/region
            reasoning_engines_response = reasoning_client.list_reasoning_engines(request=request)
            logger.info("Successfully received reasoning engines response")

            agent_count = 0
            for agent in reasoning_engines_response:
                if agent.spec.agent_framework != agent_framework:
                    logger.info(f"Agent Framework '{agent.spec.agent_framework}' is not supported. Skipping.")
                    continue
                agent_count += 1
                agent_label = agent.display_name if hasattr(agent, 'display_name') and agent.display_name else agent.name.split('/')[-1]
                logger.info(f"Found agent #{agent_count}: {agent_label} (Resource: {agent.name})")

                agents_list.append({
                    "label": agent_label,
                    "value": agent.name  # Full resource name
                })

            logger.debug(f"Total agents found: {agent_count}")

        except Exception as e:
            # If the API fails, raise an error with helpful message
            error_msg = f"Unable to fetch agents from Vertex AI Agent Engine: {str(e)}"
            logger.error(error_msg, exc_info=True)
            raise RuntimeError(error_msg) from e

        logger.info(f"Successfully retrieved {len(agents_list)} agent choices")
        return {"choices": agents_list}

    except Exception as e:
        # Re-raise the exception for Dataiku to handle
        error_msg = f"Error fetching agents: {str(e)}"
        logger.error(f"Outer exception occurred while fetching agents: {error_msg}", exc_info=True)
        raise RuntimeError(error_msg) from e
