from __future__ import annotations

import json
import os
from pathlib import Path
from typing import Tuple

import boto3
import dataiku
from dataiku import Folder
from mypy_boto3_s3 import S3Client
from mypy_boto3_s3.type_defs import DeleteTypeDef
from typing_extensions import TypedDict


class S3Credentials(TypedDict):
    region: str
    bucket: str
    root_prefix: str
    access_key_id: str
    secret_access_key: str
    session_token: str | None


class GCSCredentials(TypedDict):
    bucket: str
    project_id: str
    root_prefix: str
    access_key_id: str
    secret_access_key: str


def get_s3_credentials(folder: Folder) -> S3Credentials:
    folder_info = folder.get_info(sensitive_info=True)

    access_info = folder_info["accessInfo"]
    bucket_name = access_info["bucket"]
    root_prefix = access_info["root"]

    client = dataiku.api_client()
    project = client.get_project(folder_info["projectKey"])

    folder_params = project.get_managed_folder(folder_info["id"]).get_settings().get_raw_params()

    conn_info = client.get_connection(folder_params["connection"]).get_info()
    cred = conn_info.get_aws_credential()

    session_token = cred.get("sessionToken", None)

    s3_client = boto3.client(
        service_name="s3",
        aws_access_key_id=cred["accessKey"],
        aws_secret_access_key=cred["secretKey"],
        aws_session_token=session_token
    )
    region = s3_client.get_bucket_location(Bucket=bucket_name)["LocationConstraint"]

    return S3Credentials(
        region=region,
        bucket=bucket_name,
        root_prefix=root_prefix,
        access_key_id=cred["accessKey"],
        secret_access_key=cred["secretKey"],
        session_token=session_token
    )


def get_gcs_credentials(folder: Folder) -> GCSCredentials:
    folder_info = folder.get_info(sensitive_info=True)
    if folder_info["type"] != "GCS":
        raise Exception("Expected a GCS folder.")

    access_info = folder_info["accessInfo"]
    bucket_name = access_info["bucket"]
    project_id = access_info["projectId"]
    root_prefix = access_info["root"]

    app_secret_content = json.loads(access_info["appSecretContent"])

    return GCSCredentials(
        bucket=bucket_name,
        project_id=project_id,
        root_prefix=root_prefix,
        access_key_id=app_secret_content["private_key_id"],
        secret_access_key=app_secret_content["private_key"],
    )


def create_s3_client(folder: Folder) -> Tuple[S3Client, S3Credentials]:
    if folder.get_info()["type"] != "S3":
        raise Exception("Expected an S3 folder.")

    credentials = get_s3_credentials(folder)
    bucket_name = credentials["bucket"]
    s3_client = boto3.client(
        service_name="s3",
        aws_access_key_id=credentials["access_key_id"],
        aws_secret_access_key=credentials["secret_access_key"],
        region_name=credentials["region"],
    )

    return (s3_client, credentials)


def delete_s3_prefix(folder: Folder, relative_prefix: str) -> None:
    (s3_client, credentials) = create_s3_client(folder)
    bucket_name = credentials["bucket"]

    paginator = s3_client.get_paginator("list_objects_v2")
    for page in paginator.paginate(
        Bucket=bucket_name, Prefix=os.path.join(credentials["root_prefix"], relative_prefix)
    ):
        if "Contents" in page:
            to_delete: DeleteTypeDef = {"Objects": [{"Key": obj["Key"]} for obj in page["Contents"]]}  # type: ignore
            s3_client.delete_objects(Bucket=bucket_name, Delete=to_delete)


def upload_path_to_s3(folder: Folder, abs_path_to_download: Path) -> None:
    (s3_client, credentials) = create_s3_client(folder)
    bucket_name = credentials["bucket"]

    for root, dirs, files in os.walk(abs_path_to_download):
        for file_name in files:
            local_absolute_path = os.path.join(root, file_name)
            relative_path = os.path.relpath(local_absolute_path, abs_path_to_download)

            s3_key = os.path.join(credentials["root_prefix"], relative_path)
            s3_client.upload_file(local_absolute_path, bucket_name, s3_key)
