import logging

from flask import Blueprint, g, request

from editor.backend.utils.api import use_graph_metadata_store
from editor.backend.utils.webapp_config import webapp_config
from solutions.backend.utils import return_ok, set_auth_identifier
from solutions.graph.graph_db_instance_manager import EditorWebAppDbInstance
from solutions.graph.models import (
    FeatureUnavailableError,
    GraphDoesNotExistError,
    to_edge_group_definitions,
    to_node_group_definitions,
)
from solutions.graph.queries.params import RunLlmCypherParams
from solutions.graph.store.graph_metadata_store import GraphMetadataStore
from solutions.llm import AIQueryExecutionException, LLMCypher
from solutions.llm.logging_dataset import logging_dataset

logger = logging.getLogger(__name__)
llm = Blueprint("llm", __name__, url_prefix="/llm")


@llm.before_request
def llm_before_request():
    set_auth_identifier()


@llm.route("/generate_cypher", methods=["POST"])
@use_graph_metadata_store
def run_llm(graph_store: GraphMetadataStore):
    params = RunLlmCypherParams(**request.get_json())
    auth_identifier = g.get("auth_identifier")

    llm = webapp_config.get_llm()
    if not llm:
        raise FeatureUnavailableError("LLM must be set in webapp config")

    graph_id = params.graph_id
    if not graph_store.exists(graph_id):
        raise GraphDoesNotExistError(graph_id)

    graph_metadata = graph_store.get(graph_id)
    nodes_definitions = [
        definition
        for node_group_metadata in graph_metadata["nodes"].values()
        for definition in to_node_group_definitions(node_group_metadata)
    ]
    edges_definitions = [
        definition
        for edge_group_metadata in graph_metadata["edges"].values()
        for definition in to_edge_group_definitions(edge_group_metadata, graph_metadata)
    ]

    with EditorWebAppDbInstance(params.graph_id, readonly=True) as db_instance:
        llm_cypher = LLMCypher(
            db_instance,
            llm,
            nodes_definitions,
            edges_definitions,
        )

        try:
            data = llm_cypher.execute_llm_with_retry(
                params.query, timeout_seconds=webapp_config.db_query_timeout_seconds
            )
            if logging_dataset.is_initialized and data["success"]:
                log_uuid = logging_dataset.log_query(
                    user=auth_identifier,
                    title=data["title"],
                    llm_name=llm.llm_id,
                    question=params.query,
                    answer=data["cypher_query"],
                    graph_id=params.graph_id,
                )
                data["log_uuid"] = log_uuid
            else:
                data["log_uuid"] = None
            return return_ok(data=data)

        except AIQueryExecutionException as ex:
            logger.debug(f"AI query raised an error {params.graph_id}, '{ex.query}'.", exc_info=True)
            return return_ok(
                data={
                    "success": False,
                    "errorCode": "AI_QUERY_CYPHER_ERROR",
                    "error": str(ex),
                    "title": ex.title,
                    "cypher_query": ex.query,
                }
            )
        except Exception as ex:
            logger.exception(
                f"An unexpected error occured on AI Query {params.graph_id}, '{params.query}'.", exc_info=True
            )
            return return_ok(
                data={
                    "success": False,
                    "errorCode": "UNEXPECTED_ERROR",
                    "error": "An unexpected error occured.",
                }
            )
