# -*- coding: utf-8 -*-
import transformers
from PIL import Image
import torch
import pandas as pd
import dataiku
from utils import load_image
import io
import torchvision.transforms as T
from torchvision.transforms.functional import InterpolationMode

# Define pre-trained model and tokenizer
model_name = "OpenGVLab/InternVL2-8B"
device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

# Load the model and tokenizer
model = (
    transformers.AutoModel.from_pretrained(
        model_name, trust_remote_code=True, torch_dtype=torch.bfloat16
    )
    .eval()
    .to(device)
)
tokenizer = transformers.AutoTokenizer.from_pretrained(
    model_name, trust_remote_code=True
)

# Read input images from a Dataiku folder
folder = dataiku.Folder("Tz1CoYYW")

# Constants for image preprocessing (ImageNet mean and standard deviation)
IMAGENET_MEAN = (0.485, 0.456, 0.406)
IMAGENET_STD = (0.229, 0.224, 0.225)


# Function to load an image from the Dataiku folder
def load_image(folder, filepath):
    """
    Load an image from a Dataiku folder.

    Args:
        folder: Dataiku folder object containing the image.
        filepath: Path of the image file within the folder.

    Returns:
        PIL Image object loaded from the file.
    """
    with folder.get_download_stream(filepath) as stream:
        return Image.open(io.BytesIO(stream.read()))


# Function to build the image transformation pipeline (resize, normalize)
def build_transform(input_size):
    """
    Build a torchvision transformation pipeline to resize and normalize images.

    Args:
        input_size: Desired image size (width and height).

    Returns:
        A torchvision transformation pipeline.
    """
    MEAN, STD = IMAGENET_MEAN, IMAGENET_STD
    transform = T.Compose(
        [
            T.Lambda(lambda img: img.convert("RGB") if img.mode != "RGB" else img),
            T.Resize((input_size, input_size), interpolation=InterpolationMode.BICUBIC),
            T.ToTensor(),
            T.Normalize(mean=MEAN, std=STD),
        ]
    )
    return transform


# Function to find the closest aspect ratio for cropping and processing the image
def find_closest_aspect_ratio(aspect_ratio, target_ratios, width, height, image_size):
    """
    Find the closest aspect ratio from a list of target aspect ratios.

    Args:
        aspect_ratio: The aspect ratio of the input image.
        target_ratios: List of target aspect ratios.
        width: Width of the input image.
        height: Height of the input image.
        image_size: The desired image size for processing.

    Returns:
        The closest aspect ratio tuple (width, height).
    """
    best_ratio_diff = float("inf")
    best_ratio = (1, 1)
    area = width * height
    for ratio in target_ratios:
        target_aspect_ratio = ratio[0] / ratio[1]
        ratio_diff = abs(aspect_ratio - target_aspect_ratio)
        if ratio_diff < best_ratio_diff:
            best_ratio_diff = ratio_diff
            best_ratio = ratio
        elif ratio_diff == best_ratio_diff:
            if area > 0.5 * image_size * image_size * ratio[0] * ratio[1]:
                best_ratio = ratio
    return best_ratio


# Function to preprocess an image dynamically based on aspect ratio
def dynamic_preprocess(
    image, min_num=1, max_num=12, image_size=448, use_thumbnail=False
):
    """
    Dynamically preprocess an image by splitting it into blocks based on aspect ratio.

    Args:
        image: PIL Image object to preprocess.
        min_num: Minimum number of blocks.
        max_num: Maximum number of blocks.
        image_size: Desired image size for each block.
        use_thumbnail: Boolean to add a thumbnail version of the image if multiple blocks are generated.

    Returns:
        A list of cropped or resized image blocks.
    """
    orig_width, orig_height = image.size
    aspect_ratio = orig_width / orig_height

    # Generate target aspect ratios
    target_ratios = set(
        (i, j)
        for n in range(min_num, max_num + 1)
        for i in range(1, n + 1)
        for j in range(1, n + 1)
        if i * j <= max_num and i * j >= min_num
    )
    target_ratios = sorted(target_ratios, key=lambda x: x[0] * x[1])

    # Find the closest aspect ratio to the target
    target_aspect_ratio = find_closest_aspect_ratio(
        aspect_ratio, target_ratios, orig_width, orig_height, image_size
    )

    # Calculate the target width and height
    target_width = image_size * target_aspect_ratio[0]
    target_height = image_size * target_aspect_ratio[1]
    blocks = target_aspect_ratio[0] * target_aspect_ratio[1]

    # Resize the image to fit the target aspect ratio
    resized_img = image.resize((target_width, target_height))
    processed_images = []

    # Split the resized image into blocks
    for i in range(blocks):
        box = (
            (i % (target_width // image_size)) * image_size,
            (i // (target_width // image_size)) * image_size,
            ((i % (target_width // image_size)) + 1) * image_size,
            ((i // (target_width // image_size)) + 1) * image_size,
        )
        split_img = resized_img.crop(box)
        processed_images.append(split_img)

    # Optionally add a thumbnail version of the image
    if use_thumbnail and len(processed_images) != 1:
        thumbnail_img = image.resize((image_size, image_size))
        processed_images.append(thumbnail_img)

    return processed_images


# Function to load and preprocess an image for vision-language models
def load_image_VL(image_file, input_size=448, max_num=12):
    """
    Load and preprocess an image for vision-language models.

    Args:
        image_file: Path of the image file to load.
        input_size: Desired size for image processing.
        max_num: Maximum number of image blocks to process.

    Returns:
        A tensor containing preprocessed image blocks.
    """
    image = load_image(folder, image_file).convert("RGB")
    transform = build_transform(input_size=input_size)
    images = dynamic_preprocess(
        image, image_size=input_size, use_thumbnail=True, max_num=max_num
    )
    pixel_values = [transform(image) for image in images]
    pixel_values = torch.stack(pixel_values)
    return pixel_values

# Initialize lists for storing file names and predictions
predictions = []
filenames = []

# Generation configuration for the model
generation_config = dict(max_new_tokens=1000, do_sample=False)

# The question to ask the model for OCR
#question = "Transcribe the text from the image. Do not add any comments or additional information."

question = """You are an OCR processor. Transcribe the text given in a way that I can copy paste it in a notepad. Do not write markdown or LateX. Always provide an answer even if the picture looks low quality."""

# Loop through each file in the folder and generate predictions
for file in folder.list_paths_in_partition():
    filenames.append(file)

    # Load and preprocess the image
    pixel_values = load_image_VL(file, max_num=12).to(torch.bfloat16).cuda()

    # Generate a transcription response from the model
    response = model.chat(
        tokenizer,
        pixel_values,
        question,
        generation_config,
        history=None,
        return_history=False,
    )

    predictions.append(response)

# Create a DataFrame to store the results
df = pd.DataFrame({"file": filenames, "text": predictions})

# Save the results in a Dataiku dataset
dataiku.Dataset("results_InternVL").write_with_schema(df)
