import io
import cv2
import numpy as np
import colorsys

def load_image(folder, filename):
    """
    Load an image with cv2 from a folder
    """
    with folder.get_download_stream(filename) as stream:
        buf = io.BytesIO(stream.read())
        img = cv2.imdecode(np.frombuffer(buf.read(), np.uint8), 1)
        buf.close()
    return img

def save_image(folder, filename, img):
    """
    Save a cv2 image in a folder
    """
    is_success, buffer = cv2.imencode(".jpg", img)
    buf = io.BytesIO(buffer)
    folder.upload_data(filename, buf.getvalue())
    buf.close()

def process_annotations(annotations, color_map, size=1):
    """
    Specify bounding boxes to draw given a set of annotations
    """
    new_rectangles = []
    new_text_boxes = []
    for annotation in annotations:
        x_min, y_min, x_max, y_max = annotation["bbox"]
        x_max, y_max = int(x_min + x_max), int(y_min + y_max)
        x_min, y_min = int(x_min), int(y_min)
        color = color_map[annotation["category"]]
        # Draw bounding boxes
        new_rectangles.append(((x_min, y_min), (x_max, y_max), color, 4))

        # Add category and confidence level
        if "confidence" not in annotation:
            confidence = 0
            confidence_text = ""
        else:
            confidence = annotation["confidence"]
            confidence_text = ": " + str(round(confidence, 2))
        text = f'{annotation["category"]}{confidence_text}'
        w, h = cv2.getTextSize(text, cv2.FONT_HERSHEY_DUPLEX, size, size)[0]
        coordinates = (x_min + 4, y_min - 5)
        new_text_boxes.append((
            confidence,
            ((x_min - 3, y_min - h - 10), (x_min + w + 7, y_min), color, -1),
            (text, coordinates, cv2.FONT_HERSHEY_DUPLEX, size, (255, 255, 255), size, cv2.LINE_AA)
        ))
    return new_rectangles, new_text_boxes

def add_boxes(img, rectangles, text_boxes):
    """
    Add bounding boxes to an image
    """
    for rectangle in rectangles:
        cv2.rectangle(img, *rectangle)
    for text_box in sorted(text_boxes):
        cv2.rectangle(img, *text_box[1])
        cv2.putText(img, *text_box[2])

def get_color_map(cat2id, saturation=0.5, value=0.5):
    """
    Get a category-to-color mapping for several categories
    """
    n = len(cat2id)
    hsv = [(x/n, saturation, value) for x in range(n)]
    color_map = {}
    for k in sorted(list(cat2id)):
        color_map[k] = 255*np.array(colorsys.hsv_to_rgb(*hsv[cat2id[k]]))
    return color_map