# Projects[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#projects "Permalink to this heading")

Projects are the main unit for organising workflows within the Dataiku platform.

## Basic operations[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#basic-operations "Permalink to this heading")

This section provides common examples of how to programmatically manipulate Projects.

### Listing Projects[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#listing-projects "Permalink to this heading")

The main identifier for Projects is the *Project Key*. The following can be run to access the list of Project Keys on a Dataiku instance:

§ import dataiku

§ client = dataiku.api\_client()

§ # Get a list of Project Keys

§ project\_keys = client.list\_project\_keys()

### Handling an existing Project[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#handling-an-existing-project "Permalink to this heading")

To manipulate a Project and its associated items you first need to get its handle, in the form of a `dataikuapi.dss.project.DSSProject` object. If the Project already exists on the instance, run:

§ project = client.get\_project("CHURN")

You can also directly get a handle on the current Project you are working on:

§ project = client.get\_default\_project()

### Creating a new Project[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#creating-a-new-project "Permalink to this heading")

The following code will create a new empty Project and return its handle:

§ project = client.create\_project(project\_key="MYPROJECT",

§ name="My very own project",

§ owner="alice")

You can also duplicate an existing Project and get a handle on its copy:

§ original\_project = client.get\_project("CHURN")

§ copy\_result = original\_project.duplicate(target\_project\_key="CHURNCOPY",

§ target\_project\_name="Churn (copy)")

§ project = client.get\_project(copy\_result["CHURNCOPY"])

Finally, you can import a Project archive (zip file) and get a handle on the resulting Project:

§ archive\_path = "/path/to/archive.zip"

§ with open(archive\_path, "rb") as f:

§ import\_result = client.prepare\_project\_import(f).execute()

§ # TODO Get handle

### Accessing Project items[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#accessing-project-items "Permalink to this heading")

Once your Project handle is created, you can use it to create, list and interact with Project items:

§ # Print the names of all Datasets in the Project:

§ for d in project.list\_datasets()

§ print(d.name)

§ # Create a new empty Managed Folder:

§ folder = project.create\_managed\_folder(name="myfolder")

§ # Get a handle on a Dataset:

§ customer\_data = project.get\_dataset("customers")

### Exporting a Project[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#exporting-a-project "Permalink to this heading")

To create a Project export archive and save it locally (i.e. on the Dataiku instance server), run the following:

§ import os

§ dir\_path = "path/to/your/project/export/directory"

§ archive\_name = f"{project.project\_key}.zip"

§ with project.get\_export\_stream() as s:

§ target = os.path.join(dir\_path, archive\_name)

§ with open(target, "wb") as f:

§ for chunk in s.stream(512):

§ f.write(chunk)

### Deleting a Project[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#deleting-a-project "Permalink to this heading")

To delete a Project and all its associated objects, run the following:

§ project.delete()

Warning

While the Project’s Dataset objects will be deleted, by default the underlying data will remain. To clear the data as well, set the `clear\_managed\_datasets` argument to `True`. **The deletion operation is permanent so use this method with caution.**

## Detailed examples[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#detailed-examples "Permalink to this heading")

This section contains more advanced examples on Projects.

### Editing Project permissions[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#editing-project-permissions "Permalink to this heading")

You can programmatically add or change Group permissions for a given Project using the `set\_permissions()` method. In the following example, the ‘readers’ Group is added to the `DKU\_TSHIRTS` Project with read-only permissions:

§ import dataiku

§ PROJECT\_KEY = "DKU\_TSHIRTS"

§ GROUP = "readers"

§ client = dataiku.api\_client()

§ project = client.get\_project(PROJECT\_KEY)

§ permissions = project.get\_permissions()

§ new\_perm = {

§ "group": GROUP,

§ "admin": False,

§ "executeApp": False,

§ "exportDatasetsData": False,

§ "manageAdditionalDashboardUsers": False,

§ "manageDashboardAuthorizations": False,

§ "manageExposedElements": False,

§ "moderateDashboards": False,

§ "readDashboards": True,

§ "readProjectContent": True,

§ "runScenarios": False,

§ "shareToWorkspaces": False,

§ "writeDashboards": False,

§ "writeProjectContent": False

§ }

§ permissions["permissions"].append(new\_perm)

§ project.set\_permissions(permissions)

### Creating a Project with custom settings[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#creating-a-project-with-custom-settings "Permalink to this heading")

You can add pre-built properties to your Projects when creating them using the API. This example illustrates how to generate a Project and define the following properties:

* name

* description

* tags

* status

* checklist

First, create a helper function to generate the checklist :

§ def create\_checklist(author, items):

§ checklist = {

§ "title": "To-do list",

§ "createdOn": 0,

§ "items": []

§ }

§ for item in items:

§ checklist["items"].append({

§ "createdBy": author,

§ "createdOn": int(datetime.now().timestamp()),

§ "done": False,

§ "stateChangedOn": 0,

§ "text": item

§ })

§ return checklist

You can now write the creation function, which wraps the `create\_project()` method and returns a handle to the newly-created Project:

§ def create\_custom\_project(client,

§ project\_key,

§ name,

§ custom\_tags,

§ description,

§ checklist\_items):

§ current\_user = client.get\_auth\_info()["authIdentifier"]

§ project = client.create\_project(project\_key=project\_key,

§ name=name,

§ owner=current\_user,

§ description=description)

§ # Add tags

§ tags = project.get\_tags()

§ tags["tags"] = {k: {} for k in custom\_tags}

§ project.set\_tags(tags)

§ # Add checklist

§ metadata = project.get\_metadata()

§ metadata["checklists"]["checklists"].append(create\_checklist(author=current\_user,

§ items=checklist\_items))

§ project.set\_metadata(metadata)

§ # Set default status to "Draft"

§ settings = project.get\_settings()

§ settings.settings["projectStatus"] = "Draft"

§ settings.save()

§ return project

This is how you would call this function:

§ client = dataiku.api\_client()

§ tags = ["work-in-progress", "machine-learning", "priority-high"]

§ checklist = [

§ "Connect to data sources",

§ "Clean, aggregate and join data",

§ "Train ML model",

§ "Evaluate ML model",

§ "Deploy ML model to production"

§ ]

§ project = create\_custom\_project(client=client,

§ project\_key="MYPROJECT",

§ name="A custom Project",

§ custom\_tags=tags,

§ description="This is a cool Project",

§ checklist\_items=checklist)

### Export multiple Projects at once[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#export-multiple-projects-at-once "Permalink to this heading")

If instead of just exporting a single Project you want to generate exports several Projects in one go and store the resulting archives in a local Managed Folder, you can extend the usage of `get\_export\_stream()` with the following example:

§ import dataiku

§ import os

§ from datetime import datetime

§ PROJECT\_KEY = "BACKUP\_PROJECTS"

§ FOLDER\_NAME = "exports"

§ PROJECT\_KEYS\_TO\_EXPORT = ["FOO", "BAR"]

§ # Generate timestamp (e.g. 20221201-123000)

§ ts = datetime \

§ .now() \

§ .strftime("%Y%m%d-%H%M%S")

§ client = dataiku.api\_client()

§ project = client.get\_project(PROJECT\_KEY)

§ folder\_path = dataiku.Folder(FOLDER\_NAME) \

§ .get\_path()

§ for pkey in PROJECT\_KEYS\_TO\_EXPORT:

§ zip\_name = f"{pkey}-{ts}.zip"

§ pkey\_project = client.get\_project(pkey)

§ with pkey\_project.get\_export\_stream() as es:

§ target = os.path.join(folder\_path, zip\_name)

§ with open(target, "wb") as f:

§ for chunk in es.stream(512):

§ f.write(chunk)

## Reference documentation[¶](https://developer.dataiku.com/latest/concepts-and-examples/projects.html#reference-documentation "Permalink to this heading")

|  |  |

| --- | --- |

| `dataikuapi.dss.project.DSSProject`(client, ...) | A handle to interact with a project on the DSS instance. |
