import dataiku
import pandas as pd
import json

from dash import dcc
from dash import html
from dash.dependencies import Input, Output, State
import dash_bootstrap_components as dbc

df = dataiku.Dataset("past_answers").get_dataframe()

# Layout

STYLE_ANSWER = {
    "margin-top": "20px",
    "display": "none",
    "align-items": "flex-start",
    "height": "auto"
}

STYLE_DROPDOWN = {
    "width": "100%",
    "margin": "0 10px"
}

STYLE_CONTAINER = {
    "margin-top": "20px",
    "display": "flex",
    "justify-content": "center"
}

ok_icon_fill = html.Span(html.I(className="bi bi-emoji-smile-fill"))
nok_icon_fill = html.Span(html.I(className="bi bi-emoji-frown-fill"))

options = [
    {"label": "All answers", "value": 0},
    {"label": "Positive feedback only", "value": 1},
    {"label": "Negative feedback only", "value": -1},
]

controls = dbc.Form(
    html.Div([
        dcc.Dropdown(
            id='feedback_selected',
            options=options,
            value=0,
            style=STYLE_DROPDOWN,
            clearable=False
        )
    ],
        style=STYLE_CONTAINER
    )
)

app.title = "Visualization of past answers"
app.config.external_stylesheets = [
    dbc.themes.ZEPHYR,
    dbc.icons.BOOTSTRAP
]
app.layout = html.Div(
    [
        controls,
        html.Div(
            [
                html.Div(
                    dbc.Pagination(
                        id="pagination",
                        max_value=len(df),
                        fully_expanded=False,
                        first_last=True,
                        previous_next=True,   
                    ),
                    style=STYLE_CONTAINER
                ),
                html.Div(
                    [
                        dcc.Markdown(
                            id='question'
                        ),
                        html.A(
                            html.Div(
                                id='feedback',
                                style={"margin": "0px 5px"}
                            ),
                            href="#"
                        )
                    ],
                    style=STYLE_CONTAINER
                ),   
                dcc.Markdown(
                    id='result',
                    link_target="_blank"
                ),           
            ],
            id="output"
        ),
        html.Div(
            id="comment",
            style=STYLE_CONTAINER
        ),
        dcc.Store(
            id="data",
            storage_type="memory",
            data=df.to_json(orient='columns')
        )
    ],
    style={
        "margin": "auto",
        "text-align": "left",
        "max-width": "800px"
    }
)

# Callbacks
@app.callback(
    Output("question", "children"),
    Output("feedback", "children"),
    Output("result", "children"),
    Input("pagination", "active_page"),
    State("data", "data")
)
def display_answer(index, memory):
    i = 1 if index is None else index
    df2 = pd.DataFrame.from_dict(json.loads(memory))
    if len(df2) == 0:
        return "", "", ""
    question = f"**{df2.iloc[i-1].question}**"
    feedback = ok_icon_fill if df2.iloc[i-1].feedback == 1 else nok_icon_fill
    reply = df2.iloc[i-1].answer
    reply = f"***\n{reply}"
    return question, feedback, reply

@app.callback(
    Output("pagination", "max_value"),
    Output("pagination", "active_page"),
    Output("data", "data"),
    Output("comment", "children"),
    Output("output", "style"),
    Input("feedback_selected", "value"),
    prevent_initial_call=True,
)
def restrict_df(feedback):
    df2 = df.copy()
    if feedback in [1, -1]:
        df2 = df2[df2["feedback"] == feedback]
    if len(df2) > 0:
        return len(df2), 1, df2.to_json(orient='columns'), "", {}
    return 1, 1, {}, "No results", {"display": "none"}
