import dataiku
from dataiku.llm.agent_tools import BaseAgentTool
from agentic_insights.backend.agentic_insights_toolkit.agent import analytic_tools

class MyAgentTool(BaseAgentTool):
    """An empty interface for a code-based agent tool"""

    def __init__(self):
        pass

    def get_descriptor(self, tool):
        """
        Returns the descriptor of the tool, as a dict containing:
           - description (str)
           - inputSchema (dict, a JSON Schema representation)
        """
        return {
            "description": "Execute a given analytic tool dynamically.",
            "inputSchema" : {
                "$id": "analytic_tool_executor",
                "title": "Analytic Tool Executor",
                "type": "object",
                "properties": {
                    "analytic_tool_id": {
                        "type": "str",
                        "description": "ID of the analytic tool to execute."
                    },
                    "analysis_session_id": {
                        "type": "str",
                        "description": "Unique identifier of the analysis session."
                    },
                    "analytic_tool_args": {
                        "type": "object",
                        "description": "Arguments to pass to the analytic tool. All arguments are passed through the unique property input_args.",
                        "properties": {
                            "input_args": {
                                "type": "object",
                                "description": "Arguments to pass to the analytic tool."
                                }
                        },
                        "required": ["input_args"]
                    }
                },
                "required": ["analysis_session_id", "analytic_tool_id", "analytic_tool_args"]
            }
        }


    def invoke(self, input, trace):
        """
        Invokes the tool.

        The arguments of the tool invocation are in input["input"], a dict
        """
        input_params = input["input"]
        analytic_tool_id = input_params['analytic_tool_id']
        analysis_session_id = input_params['analysis_session_id']
        analytic_tool_args = input_params['analytic_tool_args']

        # Normalize analytic_tool_args: extract input_args or analytic_tool_args if nested, otherwise use directly
        if "input_args" in analytic_tool_args:
            input_args = analytic_tool_args["input_args"].copy()
        elif "analytic_tool_args" in analytic_tool_args:
            input_args = analytic_tool_args["analytic_tool_args"].copy()
        else:
            input_args = analytic_tool_args.copy()
        
        # Add analysis_session_id to the arguments
        input_args["analysis_session_id"] = analysis_session_id
        
        # Get the tool function dynamically
        func_name = f"{analytic_tool_id}_tool"
        func = getattr(analytic_tools, func_name, None)
        
        if func is None:
            raise ValueError(f"Analytic tool '{func_name}' not found in analytic_tools module")
        
        # Execute the tool function
        output = func(analytic_tool_args=input_args)
        
        return {
            "output": output or "",
            "sources": [],
        }