import dataiku
import logging

client = dataiku.api_client()

# Lazy import to avoid circular dependency
def _get_default_managed_dataset_connection():
    """Get default managed dataset connection from project variables."""
    from .utils import get_project_variables
    local_vars = get_project_variables(variable_type='local')
    return local_vars.get('DEFAULT_MANAGED_DATASET_CONNECTION', 'filesystem_managed')


def get_or_create_flow_zone(
    flow_zone_name,
    create_if_not_exist=False,
    tags=None,
    replace_tags=False,
    color=None,
    project=None
):
    """
    Retrieve or optionally create a Flow Zone in a Dataiku project.

    Args:
        flow_zone_name (str, required):
            Name of the Flow Zone to retrieve or create.
        create_if_not_exist (bool, optional):
            If True, creates the zone if it does not exist.
            If False and the zone does not exist, returns None. Default: False.
        tags (list[str] or str, optional):
            Tags to assign or append to the zone.
            If replace_tags=True, these tags fully replace existing ones.
            If replace_tags=False, tags are merged with existing ones (deduplicated).
        replace_tags (bool, optional):
            If True, existing tags are fully replaced by `tags`.
            If False, tags are merged. Default: False.
        color (str, optional):
            Flow Zone display color (hex or named color, depending on DSS UI conventions).
            If provided, overrides the current color of the zone.
        project (dataiku.DSSProject, optional):
            Dataiku project object. If not provided, uses the default project.

    Returns:
        dataiku.flow.DSSFlowZone or None:
            The Flow Zone object if found/created, otherwise None.
    """

    if project is None:
        project = client.get_default_project()

    flow = project.get_flow()

    # Map zone name -> zone object
    zones = flow.list_zones()
    zone_by_name = {z.name: z for z in zones}

    # Get or create zone
    if flow_zone_name in zone_by_name:
        flow_zone = zone_by_name[flow_zone_name]
    else:
        if not create_if_not_exist:
            return None
        flow_zone = flow.create_zone(name=flow_zone_name)

    # Nothing to update → return early
    if tags is None and color is None:
        return flow_zone

    # --- Settings update (tags + color in a single save) ---
    zone_settings = flow_zone.get_settings()
    raw = zone_settings.get_raw()

    # Handle tags if provided
    if tags is not None:
        # Normalize input tags to list
        if isinstance(tags, str):
            tags = [tags]

        current_tags = raw.get("tags") or []

        if replace_tags:
            # Full replacement (deduplicated)
            new_tags = list(dict.fromkeys(tags))
        else:
            # Merge existing + new, with deduplication
            new_tags = list(dict.fromkeys(current_tags + tags))

        raw["tags"] = new_tags

    # Handle color if provided
    if color is not None:
        # Depending on DSS version, the key might be "color"
        raw["color"] = color

    # Persist updates
    zone_settings.save()

    return flow_zone

def get_or_create_managed_folder(folder_name=None, folder_id=None, create_if_not_exist=False, project=None):
    """
    Retrieve or optionally create a managed folder in a Dataiku project.

    Args:
        folder_name (str, optional):
            Name of the managed folder to retrieve or create.
            Required if folder_id is not provided.
        folder_id (str, optional):
            ID of the managed folder to retrieve.
            If provided, returns the folder with this ID after verifying it exists.
        create_if_not_exist (bool, optional):
            If True, creates the folder if it does not exist (only when searching by name).
            If False and the folder does not exist, raises ValueError. Default: False.
        project (dataiku.DSSProject, optional):
            Dataiku project object. If not provided, uses the default project.

    Returns:
        dataiku.Folder:
            The managed folder object.

    Raises:
        ValueError:
            If neither folder_name nor folder_id is provided, or if the folder is not found
            and create_if_not_exist is False.
    """
    # Validate that at least one identifier is provided
    if folder_name is None and folder_id is None:
        raise ValueError("Either 'folder_name' or 'folder_id' must be provided")

    if project is None:
        project = client.get_default_project()

    # If folder_id is provided, verify it exists and return the folder object
    if folder_id is not None:
        try:
            folder = dataiku.Folder(folder_id, project_key=project.project_key)
            _ = folder.get_id()
            return folder
        except Exception:
            raise ValueError(f"No managed folder found with ID '{folder_id}'")

    # Search by folder_name
    if folder_name is None:
        raise ValueError("'folder_name' must be provided when 'folder_id' is not specified")

    all_folders = project.list_managed_folders()
    folders = [folder for folder in all_folders 
               if folder.get('name') == folder_name]

    if folders:
        # Folder exists, return it
        folder_id = folders[0]['id']
        return dataiku.Folder(folder_id, project_key=project.project_key)

    # Folder does not exist
    if not create_if_not_exist:
        raise ValueError(f"No managed folder found with name '{folder_name}'")

    # Create the folder
    folder = project.create_managed_folder(folder_name)
    return folder

def get_or_create_managed_dataset(dataset_name, default_connection=None, project=None):
    """
    Retrieve or optionally create a managed dataset in a Dataiku project.

    Args:
        dataset_name (str, required):
            Name of the managed dataset to retrieve or create.
        default_connection (str, optional):
            Connection name to use when creating a new dataset.
            If None, uses the value from project variables or defaults to "filesystem_managed".
        project (dataiku.DSSProject, optional):
            Dataiku project object. If not provided, uses the default project.

    Returns:
        dataiku.Dataset:
            The managed dataset object, either newly created or existing.

    Raises:
        ValueError:
            If dataset_name is not provided or if dataset creation fails.
    """
    if not dataset_name:
        raise ValueError("'dataset_name' must be provided")

    if project is None:
        project = client.get_default_project()

    # Lazy initialization of default_connection to avoid circular import
    if default_connection is None:
        default_connection = _get_default_managed_dataset_connection()

    # Check if dataset already exists
    dataset = project.get_dataset(dataset_name)
    if dataset.exists():
        logging.info("Dataset '%s' already exists!" % dataset_name)
        return dataset

    # Create the dataset
    logging.info("Creating new dataset '%s'..." % dataset_name)
    try:
        dataset_builder = project.new_managed_dataset(dataset_name)
        dataset_builder = dataset_builder.with_store_into(connection=default_connection)
        dataset = dataset_builder.create()
        return dataset
    except Exception as e:
        logging.exception(f"Failed to create dataset '{dataset_name}'")
        raise ValueError(f"Failed to create dataset '{dataset_name}': {str(e)}")

def find_agent(agent_name: str, project=None):
    """
    Finds the ID of the first LLM agent whose friendly name contains the given agent_name.

    Args:
        agent_name (str): Substring to search for in the agent's friendly name.
        project (optional): The Dataiku project instance. If None, uses the default project.

    Returns:
        str: The savedModelSmartId of the matched agent.

    Raises:
        ValueError: If no matching agent is found.
    """
    if project is None:
        project = client.get_default_project()
        
    matching_agents = [llm for llm in project.list_llms() if agent_name in llm['friendlyName']]
    
    if not matching_agents:
        raise ValueError(f"No agent found with name containing '{agent_name}'")

    return project.get_agent(matching_agents[0]['savedModelSmartId'])

def find_webapp(webapp_id = None, webapp_name = None, project=None):
    """
    Finds the ID of a webapp in the project matching the given name.

    Args:
        webapp_id (str, optional): ID of the webapp to search for.
        webapp_name (str, optional): Name of the webapp to search for.
        project (optional): The Dataiku project instance. If None, uses the default project.
    """
    if project is None:
        project = client.get_default_project()
        
    if webapp_name:
        all_webapps = project.list_webapps()
        found_ids = [w['id'] for w in all_webapps if w['name'] == webapp_name]
        if found_ids:
            webapp_id = found_ids[0]
    
    if webapp_id:
        webapp = project.get_webapp(webapp_id)
        return webapp
