import dataiku

from .object import get_or_create_managed_dataset, get_or_create_flow_zone
from .utils import apply_tags

client = dataiku.api_client()

def generate_code_python(input_datasets, output_datasets, python_code = None, recipe_name = None, flow_zone_name = None, tags = [], project = None):
    """
    Creates a Dataiku Python code recipe that transforms input datasets using the provided Python code.

    Args:
        input_datasets (list[str], required): List containing the names of the input datasets.
        output_datasets (list[str], required): List containing the names of the output datasets.
        python_code (str, required): Python code.
        recipe_name (str, optional): Recipe name; if not provided, an automatic name is generated based on the output dataset.
        project (dataiku.DSSProject, optional): The Dataiku project object. If not provided, defaults to the current project.
        flow_zone_name (str, optional): Name of the Flow zone where the recipe should be placed. If not provided, it will be placed in "Default" zone.
        tags (list[str], optional): List of tags to apply to the output datasets and recipe.

    Returns:
        dataikuapi.dss.recipe:  The created Python code recipe object.
    """

    if not project: project = client.get_default_project()
    
    #1- Initialize Recipe
    recipe_builder = project.new_recipe("python", name = recipe_name)
    for input_d in input_datasets:
        recipe_builder.with_input(input_d, project_key = project.project_key)
    for output_d in output_datasets:
        try:
            get_or_create_managed_dataset(dataset_name = output_d, project = project)
        except:
            pass
        recipe_builder.with_output(output_d)
    recipe = recipe_builder.build()
    
    #2- Set Python code
    recipe_settings = recipe.get_settings()
    recipe_settings.set_code(python_code)
    recipe_settings.save()
    
    #3- Move to flow zone
    if flow_zone_name:
        target_zone = get_or_create_flow_zone(flow_zone_name = flow_zone_name, project = project)
        recipe.move_to_zone(target_zone.id)
        
    #4- Tags
    if tags:
        apply_tags(dict_type_name = {"dataset": output_datasets,"recipe": [recipe.name]}, tags = tags, project = project)
    
    return recipe