import pandas as pd
import logging
import dataiku

from .common import convert_df_to_str
from ..adm.common import get_resource_folder
from ..dku_utils.utils import read_file_from_folder

client = dataiku.api_client()
project = client.get_default_project()

def explore_dataset(dataset_name, limit=5, project=project):
    """
    Explore a Dataiku dataset and return a formatted JSON-style preview and schema summary.

    This helper loads a sample of a Dataiku dataset into a Pandas DataFrame,
    extracts its schema, and returns a JSON-formatted markdown string containing:
    - The dataset name
    - The dataset schema (columns, types, meanings, etc.)
    - A preview of the first N rows (default limited if specified)

    Args:
        dataset_name (str): The name of the dataset to explore.
        limit (int, optional): Maximum number of rows to load from the dataset. If None, loads the entire dataset. Defaults to 5.
        project (dataikuapi.dss.project.DSSProject, optional): Project object, defaults to the current one.

    Returns:
        str: A markdown-formatted string containing the dataset schema and preview.
    """
    try:
        if not project:
            project = client.get_default_project()

        ds = project.get_dataset(dataset_name)

        # Load dataset as DataFrame
        df = ds.get_as_core_dataset().get_dataframe(limit=limit)
        
        schema_items_to_keep = ['name', 'type', 'meaning', 'comment']
        def filter_item(e):
            return {k:v for k,v in e.items() if (k in schema_items_to_keep) and v and not pd.isnull(v)}
        schema = ds.get_schema().get("columns", [])
        filtered_schema = [filter_item(e) for e in schema]

        # Convert to JSON for structured readability
        schema_str = convert_df_to_str(pd.DataFrame(filtered_schema))
        preview_str = convert_df_to_str(df)

        limit_info = f"first {limit}" if limit else "all available"

        # Build final formatted output
        output = (
            f"## Dataset Name: `{dataset_name}`\n\n"
            f"### Schema\n```json\n{schema_str}\n```\n\n"
            f"### Preview ({limit_info} rows)\n```json\n{preview_str}\n```"
        )

        return output

    except Exception as e:
        logging.exception(f"Failed to explore dataset '{dataset_name}'")
        raise ValueError(f"Failed to explore dataset '{dataset_name}': {e}")


def get_report_template(template_id):
    resource_folder = get_resource_folder()
    template = read_file_from_folder(
        folder = resource_folder,
        file_name = f"report_templates/{template_id}")
    return template 