import dataiku
from dataiku import pandasutils as pdu
import pandas as pd
import logging

from dku_utils.projects.project_commons import get_current_project_and_variables, get_all_project_dataset_names, get_all_project_recipe_names
from dku_utils.projects.recipes.recipe_commons import get_recipe_input_datasets, set_recipe_input_datasets
from dku_utils.projects.recipes.prepare_recipe import instantiate_prepare_recipe, reset_prepare_recipe_steps
from dku_utils.projects.security.sharing import share_object_with_name, unshare_object_with_name
from dku_utils.projects.flow_graph.flow_zones import move_dataset_in_flow_zone
from dku_utils.projects.datasets.dataset_commons import get_dataset_schema, set_dataset_schema


# Constants
OBJECT_TYPE_DATASET = "DATASET"
FLOW_ZONE_SOL_CSI = "Clinical Trials Intelligence Deliverables"
FLOW_ZONE_SOL_SDOH = "SDOH Solution Deliverables"

# Setup logging
logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')

def _get_source_dataset_details(data_prep_project, dataset_source_name_in_prep):
    """Gets schema and FQN for a source dataset."""
    source_dataset_schema = get_dataset_schema(data_prep_project, dataset_source_name_in_prep)
    source_dataset_fqn = f"{data_prep_project.project_key}.{dataset_source_name_in_prep}"
    return source_dataset_schema, source_dataset_fqn

def _create_or_update_prepare_recipe(project, recipe_name, input_dataset_fqn, output_dataset_name,
                                   connection_name, existing_recipes_names, current_project_key, client):
    """Creates a new prepare recipe or updates an existing one."""
    recipe_updated = False
    if recipe_name in existing_recipes_names:
        logger.info(f"Recipe {recipe_name} already exists. Updating input.")
        old_input_fqns = get_recipe_input_datasets(project, recipe_name)

        set_recipe_input_datasets(project, recipe_name, [input_dataset_fqn])
        logger.info(f"Recipe {recipe_name} input dataset updated to {input_dataset_fqn}")
        recipe_updated = True

        if old_input_fqns and old_input_fqns[0] != input_dataset_fqn:
            remove_shared_dataset_from_project(client, old_input_fqns[0], current_project_key)
    else:
        instantiate_prepare_recipe(project, recipe_name, input_dataset_fqn, output_dataset_name, connection_name)
        logger.info(f"New recipe {recipe_name} created with input {input_dataset_fqn} and output {output_dataset_name}")
    return project.get_recipe(recipe_name), recipe_updated

def import_dataset_from_project(source_project_key, source_dataset_name, target_output_dataset_name, flow_zone_name):
    client = dataiku.api_client()
    project, variables = get_current_project_and_variables()
    current_project_key = project.project_key

    connection_name = variables['standard']['main_connection']

    has_errors = False

    try:
        new_data_prep_project = client.get_project(source_project_key)
    except Exception as e:
        logger.error(f"Could not get data prep project {source_project_key}: {e}", exc_info=True)
        has_errors = True
        return

    current_project_datasets_names = get_all_project_dataset_names(project)
    current_project_recipe_names = get_all_project_recipe_names(project)
    logger.info(f"Importing dataset {source_dataset_name}")

    try:
        share_object_with_name(new_data_prep_project, source_dataset_name, OBJECT_TYPE_DATASET, current_project_key, True)
        logger.info(f"Shared {source_dataset_name} from {source_project_key} with {current_project_key}")

        source_dataset_schema, source_dataset_fqn = _get_source_dataset_details(new_data_prep_project, source_dataset_name)
        recipe_name = f"compute_{target_output_dataset_name}"

        recipe_obj, updated = _create_or_update_prepare_recipe(
            project, recipe_name, source_dataset_fqn, target_output_dataset_name,
            connection_name, current_project_recipe_names, current_project_key, client
        )
        if not updated and recipe_name not in current_project_recipe_names:
            current_project_recipe_names.append(recipe_name)
        if target_output_dataset_name not in current_project_datasets_names:
            current_project_datasets_names.append(target_output_dataset_name)

        if not updated:
            move_dataset_in_flow_zone(project, target_output_dataset_name, flow_zone_name)
            logger.info(f"Moved dataset {target_output_dataset_name} to flow zone {flow_zone_name}")
        
        # Run recipes
        try:
            logger.info(f"Running recipe: {recipe_name}")
            recipe_obj.run()
            logger.info(f"Recipe {recipe_name} run initiated/completed.")
        except Exception as e:
            logger.error(f"Failed to run recipe {recipe_name}: {e}", exc_info=True)
            has_errors = True

    except Exception as e:
        logger.error(f"Failed to process dataset {target_output_dataset_name}: {e}", exc_info=True)
        has_errors = True

    

    logger.info(f"Dataset {target_output_dataset_name} imported from {source_project_key} and moved to flow zone {flow_zone_name}")
    
    if has_errors:
        raise Exception("Main function completed with one or more errors. Check logs above for details.")

def remove_shared_dataset_from_project(client, previous_input_dataset_name, current_project_key):
    if not previous_input_dataset_name or '.' not in previous_input_dataset_name:
        logger.warning(f"Invalid dataset name for unsharing: {previous_input_dataset_name}")
        return
        
    old_data_prep_project_key, previous_input_dataset_name = previous_input_dataset_name.split(".")
    try:
        old_data_prep_project = client.get_project(old_data_prep_project_key)
        unshare_object_with_name(old_data_prep_project, previous_input_dataset_name, "DATASET", current_project_key)
        logger.info(f"Dataset {previous_input_dataset_name} from Project {old_data_prep_project_key} unshared!")
    except Exception as e:
        logger.warning(f"Error unsharing dataset {previous_input_dataset_name}: {str(e)}")
        pass

