import pandas as pd
import dataiku
from dku_utils.projects.datasets.dataset_commons import get_last_dataset_metrics_information
from dku_utils.projects.visual_ml.visual_ml_commons import (
    get_ml_task_and_settings,
    get_last_session_trained_model_ids,
    get_ml_task_best_model_id,
    recompute_ml_task_sampling
)

def load_features_to_drop(project, base_dataset):
    # Load metrics
    train_metrics_information = get_last_dataset_metrics_information(project, base_dataset)
    row_count = (
        int(train_metrics_information[train_metrics_information["metric_name"]=="COUNT_RECORDS"]
            ['metric_information'].values[0]['value']))

    # Get features with all null values
    count_null_information_df = (
        train_metrics_information[train_metrics_information["metric_name"] == "COUNT_NULL"])
    count_null_information_df = (
        count_null_information_df
        .assign(null_count=count_null_information_df["metric_information"].apply(lambda x: int(x["value"]))))
    features_to_rejet = list(
        count_null_information_df[count_null_information_df['null_count']==row_count]
        ['metric_column_or_scope'])
    return features_to_rejet


def update_retrain_and_deploy_model(project, base_dataset, analysis_id, ml_task_id, ml_recipe, deployed_model_id):
    ml_task, ml_task_settings = get_ml_task_and_settings(project, analysis_id, ml_task_id)
    # Load active features from ML task
    active_features = [
        feature for feature, config in ml_task_settings.get_raw()["preprocessing"]["per_feature"].items()
        if config["role"] == "INPUT"]

    # identify features with all null values
    features_to_rejet = load_features_to_drop(project, base_dataset)
    
    # Updating DESIGN features handling:
    print("Updating DESIGN features handling (model '{}') ...".format(ml_recipe))
    new_features_handling = {}
    for feature, config in ml_task_settings.get_raw()["preprocessing"]["per_feature"].items():
        if (feature in active_features) & (feature in features_to_rejet):
            config["role"] = "REJECT"
            print(f"Drop feature {feature}")
        new_features_handling[feature] = config
    ml_task_settings.get_raw()["preprocessing"]["per_feature"] = new_features_handling
    ml_task_settings.save()
    print("Model features successfully updated (model '{}') !".format(ml_recipe))

    # sampling & retrain model
    recompute_ml_task_sampling(ml_task)

    print("Starting model training...")
    ml_task.train()
    print("Model retrained !")

    # retreive the best model from the last training session form the ml task
    last_session_trained_model_ids = get_last_session_trained_model_ids(ml_task)
    last_session_best_model_id = get_ml_task_best_model_id(ml_task,
                                                           last_session_trained_model_ids,
                                                           "auc",
                                                           True)

    print("Redeploying best model in the flow ...")
    ml_task.redeploy_to_flow(model_id=last_session_best_model_id,
                             recipe_name=ml_recipe,
                             saved_model_id=deployed_model_id,
                             activate=True)
    print("Best model deployed !")
