import { defineStore } from "pinia";
import { useStudiesStore } from "./studies";
import { computed, ref, watch } from "vue";
import { isEqual } from "lodash";
import { CandidateSite, SiteScoreCardModel, SiteStudiesQuarterData } from "@/models/backend";
import { Loading } from "quasar";
import axios from "@/api/index";
import { extractLatLongFromGeoPoint } from "@/utils/geo";
import { omit } from "lodash";


export const useSitesStore = defineStore("sites", () => {
    const studies = useStudiesStore();

    const selectedSite = ref<string | null>(null);

    const sites = ref<CandidateSite[] | null>(null);

    const selectedSiteScoreCard = ref<SiteScoreCardModel | null>(null);

    const checkedSites = ref<string[]>([]);


    async function deleteSite(siteId: string) {
        if (sites.value) {
            sites.value = [...sites.value].filter((val) => val.Facility_ID !== siteId);
            checkedSites.value = [...checkedSites.value].filter((val) => val !== siteId);
            if (selectedSite.value === siteId) await setSelectedSite(null);
        }
    }

    const activeStudiesCount = computed(() => {
        if (selectedSiteScoreCard.value) {
            return String(new Set([...selectedSiteScoreCard.value.ongoing_studies_timeline_on_site].map((value) => value.NCTId)).size);
        }
        return 'NA'
    });

    const historicalStudiesCount = computed(() => {
        if (selectedSiteScoreCard.value) {
            return String(new Set([...selectedSiteScoreCard.value.historical_studies_timeline_on_site].map((value) => value.NCTId)).size);
        }
        return 'NA'
    });

    const avgNumberOfStudiesPerQuarter = computed(() => {
        if (selectedSiteScoreCard.value) {
            const periodVals: Record<string, string[]> = {};

            for (const item of selectedSiteScoreCard.value.active_studies_by_quarter_on_site) {
                if (periodVals[item.Period] != null) {
                    periodVals[item.Period].push(item.NCTId);
                } else {
                    periodVals[item.Period] = [item.NCTId];
                }
            }

            const count = Object.keys(periodVals).length;
            const sums = Object.values(periodVals).map((value) => ([...new Set(value)].length));

            return count > 0 ? String(Math.round(sums.reduce((a, b) => a + b, 0) / count)) : 'NA'
        }
        return 'NA'
    })



    function checkSite(siteId: string) {
        if (checkedSites.value.includes(siteId)) {
            checkedSites.value = [...checkedSites.value].filter((value) => value !== siteId);
        } else {
            checkedSites.value.push(siteId);
        }
    }

    watch(() => studies.finalCandidateSites, async (newVal, oldVal) => {
        if (!isEqual(newVal, oldVal)) {
            sites.value = [...newVal].filter((value) => value.Facility_ID != null);
            if (newVal.length > 0) {
                await setSelectedSite(newVal[0].Facility_ID);
            }
            checkedSites.value = [...checkedSites.value].filter((val) => {
                const siteIndex = sites.value?.findIndex((site) => site.Facility_ID === val)
                return siteIndex && siteIndex >= 0;
            })
        }
    }, { immediate: true, deep: true });



    async function setSelectedSite(siteId: string | null) {
        if (siteId == null) {
            selectedSite.value = null;
            selectedSiteScoreCard.value = null;
        } else {
            if (siteId !== selectedSite.value) {
                selectedSite.value = siteId
                try {
                    Loading.show();
                    const response = await axios.post<SiteScoreCardModel>(`/api/site/score_card`, { siteId });
                    selectedSiteScoreCard.value = response.data;
                } catch {
                    //
                } finally {
                    Loading.hide();
                }
            }
        }

    }

    const siteSdoh = computed(() => {
        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.studies_site_sdoh_on_site) {
            return selectedSiteScoreCard.value.studies_site_sdoh_on_site
        }
        return null;
    })


    const phasePieChartData = computed(() => {
        const phases: Record<string, string[]> = {}
        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.active_studies_by_quarter_on_site) {
            for (const item of selectedSiteScoreCard.value.active_studies_by_quarter_on_site) {
                const phase = item.Phase ? item.Phase : 'NA';
                if (!phases[phase]) phases[phase] = [];
                phases[phase].push(item.NCTId);
            }
        }
        const data = Object.entries(phases).map((value) => {
            const phase = value[0];
            const count = [...new Set(value[1])].length
            return { value: count, name: phase };
        })
        return {
            title: 'study_phase',
            data: data
        }
    });

    const diseaseStudiedAtSite = computed(() => {
        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.studies_mesh_conditions_on_site) {
            console.log("DEBUG SORT DATA PREVALENCE")
            console.log([...selectedSiteScoreCard.value.studies_mesh_conditions_on_site].sort((a, b) => b.prevalence - a.prevalence).slice(0, 10))
            return {
                title: 'disease_studied_at_site',
                data: selectedSiteScoreCard.value.studies_mesh_conditions_on_site.sort((a, b) => b.prevalence - a.prevalence).slice(0, 10).map((val) => ({ name: val.group_mesh_term, value: val.prevalence })).sort((a, b) => a.value - b.value),
                xAxisName: "prevalence"
            }
        }
    });

    const newStudiesPerQuarter = computed(() => {
        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.active_studies_by_quarter_on_site) {
            // keep only first period of each NTCID
            const result: Record<string, SiteStudiesQuarterData> = {}
            for (const item of selectedSiteScoreCard.value.active_studies_by_quarter_on_site) {
                const NCTId = item.NCTId;
                if (!result[NCTId] || item.Period < result[NCTId].Period) result[NCTId] = { ...item, Phase: item.Phase ? item.Phase : 'NA' };
            }
            const windowedRecords = Object.values(result);
            // create row data, xaxis periods and phases
            const phases = [...new Set(windowedRecords.map((value) => value.Phase))];
            const periods = [... new Set(windowedRecords.map((value) => value.Period))].sort();

            const rowData: number[][] = [];

            // Initialize matrix with zeros
            for (let i = 0; i < phases.length; i++) {
                rowData[i] = new Array(periods.length).fill(0);
            }

            // Fill matrix with counts
            for (const record of windowedRecords) {
                const phaseIndex = phases.indexOf(record.Phase);
                const periodIndex = periods.indexOf(record.Period);
                rowData[phaseIndex][periodIndex]++;
            }

            return {
                data: { series: phases as string[], rowData, categories: periods },
                title: 'new_studies_per_quarter'
            }

        }
        return null;
    });

    const activeStudyCountPerQuarter = computed(() => {
        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.active_studies_by_quarter_on_site) {
            const records = selectedSiteScoreCard.value.active_studies_by_quarter_on_site.map((value) => ({ ...value, Phase: value.Phase ? value.Phase : 'NA' }))
            const periods = [... new Set(records.map((value) => value.Period))].sort();
            const phases = [...new Set(records.map((value) => value.Phase))];
            const rowData: number[][] = [];

            for (let i = 0; i < phases.length; i++) {
                rowData[i] = new Array(periods.length).fill(0);
            }

            for (const record of records) {
                const phaseIndex = phases.indexOf(record.Phase);
                const periodIndex = periods.indexOf(record.Period);
                rowData[phaseIndex][periodIndex]++;
            }

            return {
                data: { series: phases as string[], rowData, categories: periods },
                title: 'active_studies_count_per_querter'
            }
        }
        return null;
    });

    const siteUsedBySponsor = computed(() => {

        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.site_used_by_sponsor) {
            const rowData: number[][] = []
            const categories = [... new Set(selectedSiteScoreCard.value.site_used_by_sponsor.map((value) => value.LeadSponsorName))].reverse()
            const groups = [... new Set(selectedSiteScoreCard.value.site_used_by_sponsor.map((value) => value.Recruiting))]

            groups.forEach((group, i) => {
                rowData[i] = new Array(categories.length).fill(0);
                categories.forEach((leadSponsor, j) => {
                    const item = selectedSiteScoreCard.value?.site_used_by_sponsor.find((value) => value.LeadSponsorName === leadSponsor && value.Recruiting === group);
                    if (item) {
                        rowData[i][j] = item.count;
                    }
                })
            });

            return {
                data: { series: groups, rowData, categories: categories },
                title: 'site_used_by_sponsor'
            }

        }
        return null;
    });

    const rawDataHistorical = computed(() => {
        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.historical_studies_timeline_on_site) {
            const rawDataDays: (string | number)[][] = [];
            const rawDataEnrollement: (string | number)[][] = [];

            rawDataDays.push(['Phase', 'StudyPeriod']);
            rawDataEnrollement.push(['Phase', 'EnrollmentCount']);
            for (const item of selectedSiteScoreCard.value.historical_studies_timeline_on_site.sort((a, b) => (a.Phase || 'NA')?.localeCompare(b.Phase || 'NA'))) {
                rawDataDays.push([item.Phase || 'NA', item.StudyPeriod || 0]);
                rawDataEnrollement.push([item.Phase || 'NA', item.EnrollmentCount || 0])
            }

            console.log({
                daysData: {
                    rawData: rawDataDays,
                    yAxisKey: 'Phase',
                    valueKey: 'StudyPeriod'
                },
                enrollementData: {
                    rawData: rawDataEnrollement,
                    yAxisKey: 'Phase',
                    valueKey: 'EnrollmentCount'
                }
            })

            return {
                daysData: {
                    rawData: rawDataDays,
                    yAxisKey: 'Phase',
                    valueKey: 'StudyPeriod'
                },
                enrollementData: {
                    rawData: rawDataEnrollement,
                    yAxisKey: 'Phase',
                    valueKey: 'EnrollmentCount'
                }
            }
        }
        return null;
    });

    const selectedSiteData = computed(() => {
        if (sites.value && selectedSite.value) {
            const siteIndex = sites.value.findIndex((val) => val.Facility_ID === selectedSite.value);
            if (siteIndex >= 0) {
                return sites.value[siteIndex];
            }
        }
        return null;
    });

    const selectedSiteCoordinates = computed(() => {
        if (selectedSiteData.value) {
            return extractLatLongFromGeoPoint(selectedSiteData.value.GeoZip);
        }
        return null;
    });

    const exportSiteList = computed(() => {
        if (sites.value && checkedSites.value) {
            const exportList = sites.value.filter((val) => {
                return checkedSites.value.includes(val.Facility_ID);
            }).map((val) => {
                return {
                    ...omit(val, 'sort_val'),
                    Similarity: (Math.round(val.Similarity.reduce((a, b) => a + b, 0) * 100) / 100).toFixed(2),
                    Study_info: JSON.stringify(val.Study_info.map((value) => {
                        return {
                            Study_id: value.length > 0 ? value[0] : 'NA',
                            Study_status: value.length > 1 ? value[1] : 'NA',
                            Site_status: value.length > 2 ? value[2] : 'NA'
                        }
                    }))
                }
            });
            return exportList;
        }
        return [];
    });


    const currentStudiesTimeline = computed(() => {
        if (selectedSiteScoreCard.value && selectedSiteScoreCard.value.ongoing_studies_timeline_on_site) {
            const data = selectedSiteScoreCard.value.ongoing_studies_timeline_on_site.map((value) => {
                return {
                    ...value,
                    Phase: value.Phase || 'NA'
                }
            }).filter((val) => val.StartDate != null && val.CompletionDate != null);

            const phases = [...new Set(data.map((value) => value.Phase))]

            const yAxisCats: Record<string, Set<string>> = {}
            const completionDates: Record<string, number[]> = {}
            const startDates: Record<string, number[]> = {}

            for (const item of data) {
                const key = `${item.Phase}_${item.NCTId}`;
                if (!yAxisCats[item.Phase]) {
                    yAxisCats[item.Phase] = new Set();
                    completionDates[item.Phase] = [];
                    startDates[item.Phase] = []
                }
                if (!(key in yAxisCats[item.Phase])) {
                    yAxisCats[item.Phase].add(key);
                    completionDates[item.Phase].push(new Date(item.CompletionDate).getTime() - new Date(item.StartDate).getTime());
                    startDates[item.Phase].push(new Date(item.StartDate).getTime());
                }
            }

            return {
                completionDates: completionDates,
                yAxisCats: yAxisCats,
                startDates: startDates
            }

        }
    })


    return {
        sites,
        selectedSite,
        checkedSites,
        checkSite,
        setSelectedSite,
        deleteSite,
        siteSdoh,
        activeStudiesCount,
        historicalStudiesCount,
        avgNumberOfStudiesPerQuarter,
        phasePieChartData,
        diseaseStudiedAtSite,
        newStudiesPerQuarter,
        activeStudyCountPerQuarter,
        siteUsedBySponsor,
        rawDataHistorical,
        selectedSiteCoordinates,
        selectedSiteData,
        exportSiteList,
        currentStudiesTimeline
    }
})