"""
Python TypedDict to Typescript Interfaces transpiler CLI.


Require Quicktype for JSON Schema to Typescript conversion: https://github.com/quicktype/quicktype
-> Install with: npm install -g quicktype

Usage:
    python transpiler.py $package_name

Custom prefixes:
    sudo ./venv/bin/python3 transpiler.py $package_name --prefixes $prefix1 $prefix2 ...

Warning:
    For dataclasses use "from pydantic.dataclasses import dataclass" instead of "from dataclasses import dataclass"
"""

from argparse import ArgumentParser
from pydantic import create_model
from uuid import uuid4
import dataclasses
import subprocess
import importlib
import inspect
import os.path
import string
import re

# Parse arguments
parser = ArgumentParser(
    description="Generate TypeScript interfaces from TypedDict python class contained in .py file."
)
parser.add_argument("input_file", help="Input Python package name")
parser.add_argument("output_file", nargs="?", help="Output TypeScript file name")
parser.add_argument(
    "--prefixes", nargs="*", help="Custom prefixes for Interfaces naming conventions"
)
args = parser.parse_args()

# Input python package
INPUT = args.input_file
if INPUT.endswith(".py"):
    INPUT = os.path.splitext(INPUT)[0]  # Remove ".py": need package name

# Patch prefixes
if args.prefixes is not None:
    # New Prefixes
    prefixes = args.prefixes if args.prefixes != [] else string.ascii_uppercase
    prefixes = ", ".join([f'"{p}"' for p in prefixes])
    # Generate path
    node_path = (
        subprocess.run(["npm", "-g", "root"], stdout=subprocess.PIPE)
        .stdout.decode("utf-8")
        .replace("\n", "")
    )
    path = node_path + "/quicktype/dist/quicktype-core/Naming.js"
    print(f"Patch: {path}")
    # Read
    text = ""
    with open(path, "r") as f:
        text = f.read()
    # Patch
    text = re.sub(
        "const funPrefixes = \[([^\]]|\n)*\]", f"const funPrefixes = [{prefixes}]", text
    )
    # Write
    with open(path, "w+") as f:
        f.write(text)

# Output TypeScript filename
OUTPUT = args.output_file if args.output_file is not None else f"{INPUT}.ts"

# Temporary filename
TMP = "__tmp__" + uuid4().hex

tmp_properties = {}
# Iterate through "TypedDict" and "Enum" classes contained in python package
for name, cls in inspect.getmembers(importlib.import_module(INPUT), inspect.isclass):
    if (
        (cls.__class__.__name__ == "_TypedDictMeta" and name != "TypedDict")
        or dataclasses.is_dataclass(cls)
        or (cls.__class__.__name__ == "EnumMeta" and name != "Enum")
    ):
        print(f"Found: {name}")
        tmp_properties[name.lower()] = (cls, ...)

# Check found classes to convert
if tmp_properties == {}:
    print("Found 0 classes to convert.")
    exit(0)

print(tmp_properties)
# Create a temporary node containing all classes
tmp = create_model(TMP, **tmp_properties)
json_model = tmp.schema_json(indent=4)

# Convert & Save as JSON scheme
with open(f"{TMP}.json", "w+") as f:
    f.write(json_model)

# Convert JSON scheme to Typescript using quicktype
os.system(f"quicktype -s schema {TMP}.json -o {TMP}.ts --just-types")

# Read tmp file
with open(f"{TMP}.ts") as f:
    text = f.read()

# Delete tmp files
os.remove(f"{TMP}.json")
os.remove(f"{TMP}.ts")

# Remove tmp node
text = "\n".join(text.split("\n")[(3 + len(tmp_properties)) :])

# Write final output file
with open(OUTPUT, "w+") as f:
    f.write(text)
