import { formatNumbers } from "./utils.ts";
export enum ChartType {
    HorizontalBar,
    VerticalBar,
    LineChart
}
function getShape(shape: "arrow-up" | "arrow-down" | "rect" | "none", color: string): string {
    switch (shape) {
        case "arrow-up":
            return `<div id="tooltip-icon"><svg width="13" height="13" viewBox="0 0 13 13" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path d="M2.67075 6.3763C2.87857 6.55814 3.19445 6.53708 3.3763 6.32926L6 3.33074V11C6 11.2761 6.22386 11.5 6.5 11.5C6.77614 11.5 7 11.2761 7 11V3.33074L9.6237 6.32926C9.80555 6.53708 10.1214 6.55814 10.3292 6.3763C10.5371 6.19445 10.5581 5.87857 10.3763 5.67075L6.88214 1.67745C6.87212 1.66561 6.86146 1.65413 6.85018 1.64308C6.75389 1.54841 6.62725 1.50005 6.5 1.5C6.37257 1.50005 6.24576 1.54854 6.14942 1.64348C6.13828 1.65441 6.12776 1.66575 6.11785 1.67746L2.62372 5.67075C2.44188 5.87857 2.46294 6.19445 2.67075 6.3763Z" fill=${color} /></svg></div>`;
        case "arrow-down":
            return `<svg width="13" height="13" viewBox="0 0 13 13" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M7 2C7 1.72386 6.77614 1.5 6.5 1.5C6.22386 1.5 6 1.72386 6 2V9.66927L3.3763 6.67075C3.19445 6.46294 2.87857 6.44188 2.67075 6.62372C2.46294 6.80556 2.44188 7.12144 2.62372 7.32926L6.11785 11.3226C6.12754 11.334 6.13783 11.3451 6.1487 11.3558C6.24513 11.4512 6.37225 11.5 6.5 11.5C6.62725 11.5 6.75389 11.4516 6.85018 11.3569C6.86146 11.3459 6.87212 11.3344 6.88214 11.3226L10.3763 7.32926C10.5581 7.12144 10.5371 6.80556 10.3292 6.62372C10.1214 6.44188 9.80555 6.46294 9.6237 6.67075L7 9.66927V2Z"
            fill="${color}"/>
          </svg>`;
        case "rect":
            return `<svg width="13" height="13" viewBox="0 0 13 13" fill="${color}" xmlns="http://www.w3.org/2000/svg">
                <rect width="13" height="13" />
            </svg>`;
        default:
            return "";
    }
}

function template(label: string, ...values: { key: string, value: string }[]): string {
    return `
      <div style="display: flex; flex-direction: column; align-items: flex-start; gap: 4px;">
          <div style="border-bottom: 1px solid #ccc; width: 100%; text-align: left; padding-bottom: 4px; color: #666666" class="bs-font-medium-2-semi-bold">
              ${label}
          </div>
          ${values.map(val => `
            <div style="display: flex; justify-content: space-between; align-items: center; width: 100%;gap: 20px;">
                <span class="dku-tiny-text" style="font-size:14px">${val.key}</span>
                <span class="dku-tiny-text" style="font-size:14px">${val.value}</span>
            </div>`).join("")}
      </div>
  `;
}

function horizontalBarTooltip(params: any): string {
    const label = params.name;
    const tooltip = params.data.tooltip;
    const upliftColor = params.data.itemStyle.color;
    const markdownOptimization = formatNumbers(tooltip.markdownOptimization);
    const strategyValue = tooltip.constantMarkdown ? formatNumbers(tooltip.constantMarkdown) : formatNumbers(tooltip.noMarkdown);
    const strategyKey = tooltip.constantMarkdown ? "Current markdown " : "No markdown ";
    const valueShape = getShape(tooltip.uplift > 0 ? "arrow-up" : tooltip.uplift < 0 ? "arrow-down" : "none", upliftColor);
    const uplift = `${valueShape} <div style="font-weight: bold; color: ${upliftColor}">${formatNumbers(tooltip.uplift)}%</div>`;

    return template(label,
        { key: "Markdown optimization", value: `<b>${markdownOptimization}</b>` },
        { key: strategyKey, value: `<b>${strategyValue}</b>` },
        { key: "Uplift", value: `<div style="display: flex; justify-content: center;">${uplift}</div>` }
    );
}
function elasticityTooltip(params: any, title: string): string {
    return template(params.seriesName,
        { key: title, value: `<b>${formatNumbers(params.data[1])}</b>` },
        { key: "Price markdown", value: `<b>${formatNumbers(params.data[0])}</b>` },
    );
}
function verticalBarTooltip(params: any, title: string): string {
    return template(params.data.name,
        { key:  `${getShape("rect", params.color)} ${params.seriesName}`, value:""},
        { key: title, value: formatNumbers(params.data.value) }
    );
}
export function tooltipOptions(chartType: ChartType, title?: string) {
    return {
        trigger: "item",
        axisPointer: {
            type: "shadow"
        },
        formatter: function (params: any) {
            switch (chartType) {
                case ChartType.HorizontalBar:
                    return horizontalBarTooltip(params);
                case ChartType.LineChart:
                    return elasticityTooltip(params, title!);
                case ChartType.VerticalBar:
                    return verticalBarTooltip(params, title!);
                default:
                    return "";
            }
        },
        backgroundColor: "#FFF",
        borderColor: "transparent",
        borderWidth: 0,
        padding: [8, 12],
        textStyle: {
            color: "black",
        },
    };
};
